package immibis.core.microblock;

import immibis.core.api.util.Dir;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.minecraft.block.Block;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;

import com.google.common.io.ByteArrayDataOutput;
import com.google.common.io.ByteStreams;

public class CoverImpl {
	
	ArrayList<Part> parts = new ArrayList<Part>();
	public BlockMicroblockContainer wrappedBlock;
	public double hollow_edge_size;
	
	public TileEntity te;
	public int partiallyDamagedPart;
	
	public CoverImpl(TileEntity te, double hes) {
		this.te = te;
		hollow_edge_size = hes;
	}
	
	public CoverImpl(TileEntity te) {
		this(te, 0.25);
	}
	
	

	public void writeToNBT(NBTTagCompound tag) {
		NBTTagList l = new NBTTagList();
		for(Part p : parts)
			l.appendTag(p.writeToNBT());
		tag.setTag("ICMP", l); // ICMP = Immibis Core Multi-Part
	}
	
	public void readFromNBT(NBTTagCompound tag) {
		parts.clear();
		NBTTagList l = tag.getTagList("ICMP");
		if(l == null)
			return;
		for(int k = 0; k < l.tagCount(); k++)
			parts.add(Part.readFromNBT(l.tagAt(k)));
	}

	public MovingObjectPosition collisionRayTrace(World world, int x, int y, int z, Vec3 src, Vec3 dst) {
		int k = 0;
		src = src.addVector(-x, -y, -z);
		dst = dst.addVector(-x, -y, -z);
		double best = dst.squareDistanceTo(src) + 1;
		Part hit = null;
		MovingObjectPosition hitInfo = null;
		int subHit = -1;
		for(Part p : parts)
		{
			AxisAlignedBB aabb = p.getBoundingBoxFromPool();
			MovingObjectPosition rt = aabb.calculateIntercept(src, dst);
			if(rt != null)
			{
				double rtdist = rt.hitVec.squareDistanceTo(src);
				if(rtdist < best)
				{
					hitInfo = rt;
					best = rtdist;
					hit = p;
					subHit = -2 - k;
				}
			}
			++k;
		}
		
		if(hit == null)
			return null;

		MovingObjectPosition pos = new MovingObjectPosition(x, y, z, hitInfo.sideHit, hitInfo.hitVec.addVector(x, y, z));
		pos.subHit = subHit;
		return pos;
	}

	public boolean addPart(Part part) {
		if(!canPlace(part.type, part.pos))
			return false;
		parts.add(part);
		return true;
	}
	
	public boolean canPlaceCentre(double size) {
		AxisAlignedBB aabb = Part.getBoundingBoxFromPool(EnumPosition.Centre, size);
		for(Part p : parts)
		{
			if(p.getBoundingBoxFromPool().intersectsWith(aabb))
				return false;
		}
		return true;
	}

	public boolean canPlace(PartType type, EnumPosition pos) {
		for(Part p : parts)
		{
			if(p.pos == pos)
				return false;
			if(p.pos.clazz == pos.clazz)
				continue;
			if(p.getBoundingBoxFromPool().intersectsWith(Part.getBoundingBoxFromPool(pos, type.size)))
				return false;
		}
		return te == null || !((ICoverableTile)te).isPlacementBlockedByTile(type, pos);
	}

	@SuppressWarnings({ "rawtypes", "unchecked" })
	public void getCollidingBoundingBoxes(World world, int x, int y, int z, AxisAlignedBB mask, List list) {
		for(Part p : parts) {
			AxisAlignedBB bb = p.getBoundingBoxFromPool().getOffsetBoundingBox(x, y, z);
			if(mask.intersectsWith(bb))
				list.add(bb);
		}
	}

	/**
	 * Returns true if tubes/cables/etc that use the centre of the block
	 * can connect through the specified side - ie if it is not blocked by a non-hollow cover.
	 */
	public boolean isSideOpen(int side) {
		return !isPositionOccupied(EnumPosition.getFacePosition(side), true);
	}

	public boolean isSideOpen(int myx, int myy, int myz, int x, int y, int z) {
		if(x < myx) return isSideOpen(Dir.NX);
		if(x > myx) return isSideOpen(Dir.PX);
		if(y < myy) return isSideOpen(Dir.NY);
		if(y > myy) return isSideOpen(Dir.PY);
		if(z < myz) return isSideOpen(Dir.NZ);
		if(z > myz) return isSideOpen(Dir.PZ);
		throw new IllegalArgumentException("no direction given (start = end)");
	}
	
	public byte[] writeDescriptionBytes() {
		ByteArrayDataOutput o = ByteStreams.newDataOutput(6 + parts.size()*3);
		try {
			writeDescriptionPacket(o);
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
		return o.toByteArray();
	}
	
	public void readDescriptionBytes(byte[] data, int start) {
		try {
			readDescriptionPacket(ByteStreams.newDataInput(data, start));
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
	}

	public void writeDescriptionPacket(DataOutput data) throws IOException {
		data.writeShort(wrappedBlock == null ? 0 : wrappedBlock.blockID);
		data.writeShort(parts.size());
		data.writeShort(partiallyDamagedPart);
		//data.writeInt(te.xCoord);
		//data.writeInt(te.yCoord);
		//data.writeInt(te.zCoord);
		for(Part pt : parts) {
			data.writeByte(pt.pos.ordinal());
			data.writeShort(pt.type.id);
		}
	}
	
	public void readDescriptionPacket(DataInput data) throws IOException {
		int wrappedID = data.readShort();
		wrappedBlock = (wrappedID == 0 ? null : (BlockMicroblockContainer)Block.blocksList[wrappedID]);
		int ncovers = data.readShort();
		partiallyDamagedPart = data.readShort();
		//int x = data.readInt();
		//int y = data.readInt();
		//int z = data.readInt();
		//if(x != te.xCoord || y != te.yCoord || z != te.zCoord)
		//	throw new IOException("Coordinates don't match");
		parts.clear();
		for(int k = 0; k < ncovers; k++)
		{
			EnumPosition pos = EnumPosition.values()[data.readByte()];
			int type = data.readShort() & 65535;
			parts.add(new Part(CoverSystemProxy.parts.get(type), pos));
		}
		
		te.worldObj.markBlockForUpdate(te.xCoord, te.yCoord, te.zCoord);
	}
	/* $endif$ */

	public void copyPartsTo(CoverImpl other) {
		for(Part p : parts)
			other.addPart(p);
	}

	/**
	 * Removes the part at the given coordinates.
	 * Returns the items dropped, or null (if no items should be dropped)
	 */
	public List<ItemStack> removePartByPlayer(World w, EntityPlayer ply, int x, int y, int z, int subhit) {
		
		TileEntity te = w.getBlockTileEntity(x, y, z);
		if(te == null || !(te instanceof ICoverableTile)) {
			w.setBlockWithNotify(x, y, z, 0);
			return null;
		}
		
		if(parts.size() == 0 && wrappedBlock == null) {
			w.setBlockWithNotify(x, y, z, 0);
			return null;
		}
		
		if(subhit >= 0)
		{
			if(wrappedBlock != null)
			{
				List<ItemStack> rv = wrappedBlock.removeBlockByPlayerMultipart(w, ply, x, y, z, subhit);
				return rv;
			} else
				return null;
		}
		else if(subhit == -1)
			return null;
		
		int partIndex = -2 - subhit;
		
		CoverImpl cover = ((ICoverableTile)te).getCoverImpl();
		
		if(partIndex >= cover.parts.size())
			return null;
		
		Part p = cover.parts.remove(partIndex);
		if(p == null)
			return null;
		
		ItemStack drop = new ItemStack(CoverSystemProxy.blockMultipart.blockID, 1, p.type.id);
		
		if(cover.parts.size() == 0 && wrappedBlock == null)
			w.setBlock(x, y, z, 0);
		else
			w.markBlockForUpdate(x, y, z);
		
		return Collections.singletonList(drop);
	}
	
	/**
	 * Returns true if there is a part in the specified position.
	 */
	public boolean isPositionOccupied(EnumPosition pos) {
		if(te != null && ((ICoverableTile)te).isPositionOccupiedByTile(pos))
			return true;
		for(Part p : parts)
			if(p.pos == pos)
				return true;
		return false;
	}
	
	/**
	 * Returns true if there is a part in the specified position.
	 * If ignoreHollowPanels is true, hollow panels will be ignored.
	 */
	public boolean isPositionOccupied(EnumPosition pos, boolean ignoreHollowPanels) {
		if(te != null && ((ICoverableTile)te).isPositionOccupiedByTile(pos))
			return true;
		for(Part p : parts)
			if(p.pos == pos && (!ignoreHollowPanels || p.type.clazz != EnumPartClass.HollowPanel))
				return true;
		return false;
	}

	/**
	 * Checks if an edge is unused (for example, if wire connections can connect around corners through it)
	 * Returns the same result if face1 and face2 are swapped.
	 * 
	 * @param face1 One face bordering the edge to check.
	 * @param face2 The other face bordering the edge to check.
	 * @return True if the edge is unused.
	 */
	public boolean isEdgeOpen(int face1, int face2) {
		return !isPositionOccupied(EnumPosition.getEdgePosition(face1, face2)) && isSideOpen(face1) && isSideOpen(face2);
	}
}