package immibis.microblocks.coremod;

import immibis.core.api.APILocator;
import immibis.core.api.microblock.IMicroblockCoverSystem;
import immibis.core.api.microblock.IMicroblockSupporterTile;
import immibis.core.api.microblock.IMicroblockSystem;
import immibis.core.api.multipart.ICoverSystem;
import immibis.core.api.multipart.IMultipartTile;
import immibis.core.multipart.BlockMultipartBase;
import immibis.microblocks.PacketMicroblockContainerDescription;
import immibis.microblocks.PacketMicroblockDescriptionWithWrapping;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.minecraft.block.Block;
import net.minecraft.client.renderer.RenderBlocks;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.packet.Packet;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.MovingObjectPosition;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeDirection;

import org.objectweb.asm.*;
import org.objectweb.asm.util.CheckClassAdapter;

import com.google.common.collect.ObjectArrays;

import cpw.mods.fml.relauncher.FMLRelauncher;
import cpw.mods.fml.relauncher.IClassTransformer;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

public class MicroblockSupporterTransformer implements IClassTransformer {
	public static Set<String> blockClasses = new HashSet<String>();
	public static Set<String> tileClasses = new HashSet<String>();
	
	private static final boolean MCP = MicroblocksCoreMod.MCP;
	
	private static final String AXISALIGNEDBB = MCP ? "net/minecraft/util/AxisAlignedBB" : "aoe";
	private static final String BLOCK = MCP ? "net/minecraft/block/Block" : "amq";
	private static final String WORLD = MCP ? "net/minecraft/world/World" : "yc";
	private static final String ENTITY = MCP ? "net/minecraft/entity/Entity" : "lq";
	private static final String TILEENTITY = MCP ? "net/minecraft/tileentity/TileEntity" : "any";
	private static final String IBLOCKACCESS = MCP ? "net/minecraft/world/IBlockAccess" : "ym";
	private static final String ENTITYPLAYER = MCP ? "net/minecraft/entity/player/EntityPlayer" : "qx";
	private static final String RENDERBLOCKS = MCP ? "net/minecraft/client/renderer/RenderBlocks" : "bbb";
	private static final String PACKET = MCP ? "net/minecraft/network/packet/Packet" : "ef";
	private static final String NBTTAGCOMPOUND = MCP ? "net/minecraft/nbt/NBTTagCompound" : "bq";
	private static final String MOVINGOBJECTPOSITION = MCP ? "net/minecraft/util/MovingObjectPosition" : "aoh";
	private static final String VEC3 = MCP ? "net/minecraft/util/Vec3" : "aoj";
	private static final String ITEMSTACK = MCP ? "net/minecraft/item/ItemStack" : "ur";
	
	private static final String BLOCK_ADDCOLLIDINGBLOCKTOLIST_NAME = MCP ? "addCollidingBlockToList" : "a";
	private static final String BLOCK_ADDCOLLIDINGBLOCKTOLIST_DESC = "(L"+WORLD+";IIIL"+AXISALIGNEDBB+";Ljava/util/List;L"+ENTITY+";)V";
	
	private static final String IBLOCKACCESS_GETBLOCKTILEENTITY_NAME = MCP ? "getBlockTileEntity" : "q";
	private static final String IBLOCKACCESS_GETBLOCKTILEENTITY_DESC = "(III)L" + TILEENTITY + ";";
	
	private static final String BLOCK_ONBLOCKCLICKED_NAME = MCP ? "onBlockClicked" : "a";
	private static final String BLOCK_ONBLOCKCLICKED_DESC = "(L"+WORLD+";IIIL"+ENTITYPLAYER+";)V";
	
	private static final String BLOCK_REMOVEBLOCKBYPLAYER_NAME = "removeBlockByPlayer"; // Forge method, not obfuscated
	private static final String BLOCK_REMOVEBLOCKBYPLAYER_DESC = "(L"+WORLD+";L"+ENTITYPLAYER+";III)Z";
	private static final String BLOCK_REMOVEBLOCKBYPLAYER_REPLACEMENT = "removeBlockByPlayer_ImmibisMicroblockTransformer";
	
	private static final String BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_NAME = MCP ? "getPlayerRelativeBlockHardness" : "a";
	private static final String BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_DESC = "(L" + ENTITYPLAYER + ";L" + WORLD + ";III)F";
	private static final String BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_REPLACEMENT = "getPlayerRelativeBlockHardness_ImmibisMicroblockTransformer";
	
	//private static final String BLOCK_GETBLOCKDROPPED_NAME = "getBlockDropped"; // Forge method, not obfuscated
	//private static final String BLOCK_GETBLOCKDROPPED_DESC = "(L"+WORLD+";IIIII)Ljava/util/ArrayList;";
	//private static final String BLOCK_GETBLOCKDROPPED_REPLACEMENT = "getBlockDropped_ImmibisMicroblockTransformer";
	
	private static final String BLOCK_GETRENDERTYPE_NAME = MCP ? "getRenderType" : "d";
	private static final String BLOCK_GETRENDERTYPE_DESC = "()I";
	
	private static final String TILEENTITY_GETDESCRIPTIONPACKET_NAME = MCP ? "getDescriptionPacket" : "l";
	private static final String TILEENTITY_GETDESCRIPTIONPACKET_DESC = "()L"+PACKET+";";
	
	private static final String TILEENTITY_WRITETONBT_NAME = MCP ? "writeToNBT" : "b";
	private static final String TILEENTITY_WRITETONBT_DESC = "(L" + NBTTAGCOMPOUND + ";)V";
	
	private static final String TILEENTITY_READFROMNBT_NAME = MCP ? "readFromNBT" : "a";
	private static final String TILEENTITY_READFROMNBT_DESC = "(L" + NBTTAGCOMPOUND + ";)V";
	
	private static final String BLOCK_COLLISIONRAYTRACE_NAME = MCP ? "collisionRayTrace" : "a";
	private static final String BLOCK_COLLISIONRAYTRACE_DESC = "(L" + WORLD + ";IIIL" + VEC3 + ";L" + VEC3 + ";)L" + MOVINGOBJECTPOSITION + ";";
	private static final String BLOCK_COLLISIONRAYTRACE_REPLACEMENT = "collisionRayTrace_ImmibisMicroblockTransformer";
	
	private static final String BLOCK_GETPICKBLOCK_NAME = "getPickBlock"; // Forge method
	private static final String BLOCK_GETPICKBLOCK_DESC = "(L" + MOVINGOBJECTPOSITION + ";L" + WORLD + ";III)L" + ITEMSTACK + ";";
	
	private static final String BLOCK_ISBLOCKSOLIDONSIDE_NAME = "isBlockSolidOnSide"; // Forge method
	private static final String BLOCK_ISBLOCKSOLIDONSIDE_DESC = "(L" + WORLD + ";IIILnet/minecraftforge/common/ForgeDirection;)Z";
	
	private static final String BLOCK_DROPBLOCKASITEMWITHCHANCE_NAME = MCP ? "dropBlockAsItemWithChance" : "a";
	private static final String BLOCK_DROPBLOCKASITEMWITHCHANCE_DESC = "(L" + WORLD + ";IIIIFI)V";
	
	private static class ClassVisitorBase extends ClassVisitor {
		public ClassVisitorBase(int api, ClassVisitor parent) {
			super(api, parent);
		}
		
		public void generateAndTransformMethod(ClassVisitor unused, String superclass, int access, String name, String desc) {
			Type methodType = Type.getMethodType(desc);
			
			MethodVisitor mv = visitMethod(access, name, desc, null, new String[0]);
			
			mv.visitCode();
			
			// push 'this'
			if((access & Opcodes.ACC_STATIC) == 0)
				mv.visitVarInsn(Opcodes.ALOAD, 0);
			
			// push arguments
			int stackPos = 1;
			for(Type pt : methodType.getArgumentTypes()) {
				switch(pt.getSort()) {
				case Type.ARRAY: case Type.OBJECT:
					mv.visitVarInsn(Opcodes.ALOAD, stackPos++);
					break;
				case Type.BOOLEAN: case Type.BYTE: case Type.CHAR: case Type.INT: case Type.SHORT:
					mv.visitVarInsn(Opcodes.ILOAD, stackPos++);
					break;
				case Type.DOUBLE:
					mv.visitVarInsn(Opcodes.DLOAD, stackPos);
					stackPos += 2;
					break;
				case Type.FLOAT:
					mv.visitVarInsn(Opcodes.FLOAD, stackPos++);
					break;
				case Type.LONG:
					mv.visitVarInsn(Opcodes.LLOAD, stackPos);
					stackPos += 2;
					break;
				}
			}
			
			// call superclass method
			mv.visitMethodInsn(Opcodes.INVOKESPECIAL, superclass, name, desc);
			
			// return result
			switch(methodType.getReturnType().getSort()) {
			case Type.ARRAY: case Type.OBJECT:
				mv.visitInsn(Opcodes.ARETURN);
				break;
			case Type.BOOLEAN: case Type.BYTE: case Type.CHAR: case Type.INT: case Type.SHORT:
				mv.visitInsn(Opcodes.IRETURN);
				break;
			case Type.DOUBLE:
				mv.visitInsn(Opcodes.DRETURN);
				break;
			case Type.FLOAT:
				mv.visitInsn(Opcodes.FRETURN);
				break;
			case Type.LONG:
				mv.visitInsn(Opcodes.LRETURN);
				break;
			case Type.VOID:
				mv.visitInsn(Opcodes.RETURN);
				break;
			}
			
			int nArgs = methodType.getArgumentsAndReturnSizes() >> 2;
			int nRets = methodType.getArgumentsAndReturnSizes() & 3;
			mv.visitMaxs(nArgs + 1, Math.max(nArgs + 1, nRets));
			mv.visitEnd();
		}
	}
	
	// This basically merges the code from BlockMultipartBase into any Block class.
	// See BlockMultipartBase for the code it adds (or at least the code it's supposed to add).
	private static class BlockTransformerVisitor extends ClassVisitorBase {
		public BlockTransformerVisitor(ClassVisitor parent) {
			super(Opcodes.ASM4, parent);
		}
		
		public static class TransformMethodVisitor_getRenderType extends MethodVisitor {
			public TransformMethodVisitor_getRenderType(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			@Override
			public void visitInsn(int opcode) {
				if(opcode == Opcodes.IRETURN)
					super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "getRenderTypeStatic", "(I)I");
				super.visitInsn(opcode);
			}
		}
		
		public static class TransformMethodVisitor_getTextureFile extends MethodVisitor {
			public TransformMethodVisitor_getTextureFile(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			@Override
			public void visitInsn(int opcode) {
				if(opcode == Opcodes.ARETURN)
					super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "getTextureFileStatic", "(Ljava/lang/String;)Ljava/lang/String;");
				super.visitInsn(opcode);
			}
		}
		
		public static class TransformMethodVisitor_onBlockClicked extends MethodVisitor {
			
			public TransformMethodVisitor_onBlockClicked(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			/*
			 * Add at start:
			 * 
			 * BlockMultipartBase.onBlockClickedStatic(par1World, par5Player);
			 */
			
			@Override
			public void visitCode() {
				super.visitCode();
				
				super.visitVarInsn(Opcodes.ALOAD, 1);
				super.visitVarInsn(Opcodes.ALOAD, 5);
				super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "onBlockClickedStatic", "(L"+WORLD+";L"+ENTITYPLAYER+";)V");
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 2)
					maxStack = 2;
				super.visitMaxs(maxStack, maxLocals);
			}
		}
		
		public static class TransformMethodVisitor_getPickBlock extends MethodVisitor {
			
			public TransformMethodVisitor_getPickBlock(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			/*
			 * Add at start:
			 * 
			 * ItemStack temp = Util.pickPart(par1MovingObjectPosition, par2World, par3X, par4Y, par5Z);
			 * if(temp != null)
			 * 	   return temp;
			 */
			
			@Override
			public void visitCode() {
				super.visitCode();
				
				Label ifNull = new Label();
				
				super.visitVarInsn(Opcodes.ALOAD, 1);
				super.visitVarInsn(Opcodes.ALOAD, 2);
				super.visitVarInsn(Opcodes.ILOAD, 3);
				super.visitVarInsn(Opcodes.ILOAD, 4);
				super.visitVarInsn(Opcodes.ILOAD, 5);
				super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "pickPart", "(L"+MOVINGOBJECTPOSITION+";L"+WORLD+";III)L"+ITEMSTACK+";");
				super.visitInsn(Opcodes.DUP);
				super.visitJumpInsn(Opcodes.IFNULL, ifNull);
				super.visitInsn(Opcodes.ARETURN);
				
				super.visitLabel(ifNull);
				super.visitFrame(Opcodes.F_SAME1, 0, null, 1, new Object[] {ITEMSTACK});
				super.visitInsn(Opcodes.POP);
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 5)
					maxStack = 5;
				super.visitMaxs(maxStack, maxLocals);
			}
		}
		
		public static class TransformMethodVisitor_isBlockSolidOnSide extends MethodVisitor {
			
			public TransformMethodVisitor_isBlockSolidOnSide(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			/*
			 * Add at start:
			 * 
			 * if(Util.isSolidOnSide(par1World, par2X, par3Y, par4Z, par5ForgeDirection))
			 *     return true;
			 */
			
			@Override
			public void visitCode() {
				super.visitCode();
				
				Label ifFalse = new Label();
				
				super.visitVarInsn(Opcodes.ALOAD, 1);
				super.visitVarInsn(Opcodes.ILOAD, 2);
				super.visitVarInsn(Opcodes.ILOAD, 3);
				super.visitVarInsn(Opcodes.ILOAD, 4);
				super.visitVarInsn(Opcodes.ALOAD, 5);
				super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "isSolidOnSide", "(L"+WORLD+";IIILnet/minecraftforge/common/ForgeDirection;)Z");
				super.visitJumpInsn(Opcodes.IFEQ, ifFalse);
				super.visitInsn(Opcodes.ICONST_1);
				super.visitInsn(Opcodes.IRETURN);
				
				super.visitLabel(ifFalse);
				super.visitFrame(Opcodes.F_SAME, 0, null, 0, null);
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 5)
					maxStack = 5;
				super.visitMaxs(maxStack, maxLocals);
			}
		}
		
		public static class TransformMethodVisitor_dropBlockAsItemWithChance extends MethodVisitor {
			public TransformMethodVisitor_dropBlockAsItemWithChance(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			/*
			 * Add at start:
			 * 
			 * if(Util.dropBlockAsItemWithChance(par1World, par2X, par3Y, par4Z, par6Float))
			 *     return;
			 */
			@Override
			public void visitCode() {
				super.visitCode();
				
				super.visitVarInsn(Opcodes.ALOAD, 1);
				super.visitVarInsn(Opcodes.ILOAD, 2);
				super.visitVarInsn(Opcodes.ILOAD, 3);
				super.visitVarInsn(Opcodes.ILOAD, 4);
				super.visitVarInsn(Opcodes.FLOAD, 6);
				super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "dropBlockAsItemWithChance", "(L"+WORLD+";IIIF)Z");
				
				Label ifFalse = new Label();
				
				super.visitJumpInsn(Opcodes.IFEQ, ifFalse);
				super.visitInsn(Opcodes.RETURN);
				super.visitLabel(ifFalse);
				super.visitFrame(Opcodes.F_SAME, 0, null, 0, null);
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 5)
					maxStack = 5;
				
				super.visitMaxs(maxStack, maxLocals);
			}
		}

		public static class TransformMethodVisitor_addCollidingBlockToList extends MethodVisitor {
			
			public TransformMethodVisitor_addCollidingBlockToList(MethodVisitor mv) {
				super(Opcodes.ASM4, mv);
			}
			
			/*
			 * Add at start:
			 * 
			 * IMultipartTile te = ((IMultipartTile)par1World.getBlockTileEntity(par2X, par3Y, par4Z));
			 * ICoverSystem ci = te.getCoverSystem();
			 * if(ci != null)
			 * 		ci.getCollidingBoundingBoxes(par5Mask, par6List);
			 */

			@Override
			public void visitCode() {
				super.visitCode();
				
				Label isNull = new Label(), isNotNull = new Label();
				
				// get the ICoverSystem
				super.visitVarInsn(Opcodes.ALOAD, 1); // par1World
				super.visitVarInsn(Opcodes.ILOAD, 2); // par2X
				super.visitVarInsn(Opcodes.ILOAD, 3); // par3Y
				super.visitVarInsn(Opcodes.ILOAD, 4); // par4Z
				super.visitMethodInsn(Opcodes.INVOKEINTERFACE, IBLOCKACCESS, IBLOCKACCESS_GETBLOCKTILEENTITY_NAME, IBLOCKACCESS_GETBLOCKTILEENTITY_DESC);
				super.visitTypeInsn(Opcodes.CHECKCAST, "immibis/core/api/multipart/IMultipartTile");
				super.visitMethodInsn(Opcodes.INVOKEINTERFACE, "immibis/core/api/multipart/IMultipartTile", "getCoverSystem", "()Limmibis/core/api/multipart/ICoverSystem;");
				
				// skip if it's null
				super.visitInsn(Opcodes.DUP);
				super.visitJumpInsn(Opcodes.IFNULL, isNull);
				
				super.visitVarInsn(Opcodes.ALOAD, 5); // par5Mask
				super.visitVarInsn(Opcodes.ALOAD, 6); // par6List
				super.visitMethodInsn(Opcodes.INVOKEINTERFACE, "immibis/core/api/multipart/IPartContainer", "getCollidingBoundingBoxes", "(L"+AXISALIGNEDBB+";Ljava/util/List;)V");
				super.visitJumpInsn(Opcodes.GOTO, isNotNull);
				
				super.visitLabel(isNull);
				super.visitFrame(Opcodes.F_SAME1, 0, null, 1, new Object[] {"immibis/core/api/multipart/ICoverSystem"});
				// drop extra copy of cover system from stack, if it was null
				super.visitInsn(Opcodes.POP);
				
				super.visitLabel(isNotNull);
				super.visitFrame(Opcodes.F_SAME, 0, null, 0, null);
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 4)
					maxStack = 4;
				super.visitMaxs(maxStack, maxLocals);
			}
		}
		
		public String className, superclass;
		
		public boolean saw_addCollidingBlockToList;
		public boolean saw_removeBlockByPlayer;
		public boolean saw_onBlockClicked;
		public boolean saw_getPlayerRelativeBlockHardness;
		//public boolean saw_getBlockDropped;
		public boolean saw_getRenderType;
		public boolean saw_getTextureFile;
		public boolean saw_collisionRayTrace;
		public boolean saw_getPickBlock;
		public boolean saw_isBlockSolidOnSide;
		public boolean saw_dropBlockAsItemWithChance;
		
		@Override
		public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
			className = name;
			superclass = superName;
			super.visit(version, access, name, signature, superName, interfaces);
		}
		
		@Override
		public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions) {
			
			if(name.equals(BLOCK_REMOVEBLOCKBYPLAYER_NAME) && desc.equals(BLOCK_REMOVEBLOCKBYPLAYER_DESC)) {
				name = BLOCK_REMOVEBLOCKBYPLAYER_REPLACEMENT;
				saw_removeBlockByPlayer = true;
			}
			
			if(name.equals(BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_NAME) && desc.equals(BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_DESC)) {
				name = BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_REPLACEMENT;
				saw_getPlayerRelativeBlockHardness = true;
			}
			
			/*if(name.equals(BLOCK_GETBLOCKDROPPED_NAME) && desc.equals(BLOCK_GETBLOCKDROPPED_DESC)) {
				name = BLOCK_GETBLOCKDROPPED_REPLACEMENT;
				saw_getBlockDropped = true;
			}*/
			
			if(name.equals(BLOCK_COLLISIONRAYTRACE_NAME) && desc.equals(BLOCK_COLLISIONRAYTRACE_DESC)) {
				name = BLOCK_COLLISIONRAYTRACE_REPLACEMENT;
				saw_collisionRayTrace = true;
			}
			
			MethodVisitor mv = super.visitMethod(access, name, desc, signature, exceptions);
			
			if(name.equals(BLOCK_ADDCOLLIDINGBLOCKTOLIST_NAME) && desc.equals(BLOCK_ADDCOLLIDINGBLOCKTOLIST_DESC)) {
				saw_addCollidingBlockToList = true;
				mv = new TransformMethodVisitor_addCollidingBlockToList(mv);
			}
			
			if(name.equals(BLOCK_ONBLOCKCLICKED_NAME) && desc.equals(BLOCK_ONBLOCKCLICKED_DESC)) {
				saw_onBlockClicked = true;
				mv = new TransformMethodVisitor_onBlockClicked(mv);
			}
			
			if(name.equals(BLOCK_GETRENDERTYPE_NAME) && desc.equals(BLOCK_GETRENDERTYPE_DESC)) {
				saw_getRenderType = true;
				mv = new TransformMethodVisitor_getRenderType(mv);
			}
			
			if(name.equals("getTextureFile") && desc.equals("()Ljava/lang/String;")) {
				saw_getTextureFile = true;
				mv = new TransformMethodVisitor_getTextureFile(mv);
			}
			
			if(name.equals(BLOCK_GETPICKBLOCK_NAME) && desc.equals(BLOCK_GETPICKBLOCK_DESC)) {
				saw_getPickBlock = true;
				mv = new TransformMethodVisitor_getPickBlock(mv);
			}
			
			if(name.equals(BLOCK_ISBLOCKSOLIDONSIDE_NAME) && desc.equals(BLOCK_ISBLOCKSOLIDONSIDE_DESC)) {
				saw_isBlockSolidOnSide = true;
				mv = new TransformMethodVisitor_isBlockSolidOnSide(mv);
			}
			
			if(name.equals(BLOCK_DROPBLOCKASITEMWITHCHANCE_NAME) && desc.equals(BLOCK_DROPBLOCKASITEMWITHCHANCE_DESC)) {
				saw_dropBlockAsItemWithChance = true;
				mv = new TransformMethodVisitor_dropBlockAsItemWithChance(mv);
			}
			
			return mv;
		}
		
		@Override
		public void visitEnd() {
			
			if(!saw_addCollidingBlockToList)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC ,BLOCK_ADDCOLLIDINGBLOCKTOLIST_NAME, BLOCK_ADDCOLLIDINGBLOCKTOLIST_DESC);
			
			if(!saw_removeBlockByPlayer)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_REMOVEBLOCKBYPLAYER_NAME, BLOCK_REMOVEBLOCKBYPLAYER_DESC);
			
			if(!saw_onBlockClicked)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_ONBLOCKCLICKED_NAME, BLOCK_ONBLOCKCLICKED_DESC);
			
			if(!saw_getPlayerRelativeBlockHardness)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_NAME, BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_DESC);
			
			//if(!saw_getBlockDropped)
			//	generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_GETBLOCKDROPPED_NAME, BLOCK_GETBLOCKDROPPED_DESC);
			
			if(!saw_getRenderType)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_GETRENDERTYPE_NAME, BLOCK_GETRENDERTYPE_DESC);
			
			if(!saw_getTextureFile)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, "getTextureFile", "()Ljava/lang/String;");
			
			if(!saw_collisionRayTrace)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_COLLISIONRAYTRACE_NAME, BLOCK_COLLISIONRAYTRACE_DESC);
			
			if(!saw_getPickBlock)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_GETPICKBLOCK_NAME, BLOCK_GETPICKBLOCK_DESC);
			
			if(!saw_isBlockSolidOnSide)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_ISBLOCKSOLIDONSIDE_NAME, BLOCK_ISBLOCKSOLIDONSIDE_DESC);
			
			if(!saw_dropBlockAsItemWithChance)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, BLOCK_DROPBLOCKASITEMWITHCHANCE_NAME, BLOCK_DROPBLOCKASITEMWITHCHANCE_DESC);
			
			{
				/* public boolean removeBlockByPlayer(World w, EntityPlayer ply, int x, int y, int z) {
					return BlockMultipartBase.removeBlockByPlayerStatic(w, ply, x, y, z);
				} */
				
				MethodVisitor mv = super.visitMethod(Opcodes.ACC_PUBLIC, BLOCK_REMOVEBLOCKBYPLAYER_NAME, BLOCK_REMOVEBLOCKBYPLAYER_DESC, null, new String[0]);
				mv.visitCode();
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				mv.visitVarInsn(Opcodes.ALOAD, 2);
				mv.visitVarInsn(Opcodes.ILOAD, 3);
				mv.visitVarInsn(Opcodes.ILOAD, 4);
				mv.visitVarInsn(Opcodes.ILOAD, 5);
				mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "removeBlockByPlayerStatic", BLOCK_REMOVEBLOCKBYPLAYER_DESC);
				mv.visitInsn(Opcodes.IRETURN);
				mv.visitMaxs(5, 6);
				mv.visitEnd();
			}
			
			{
				/* public float getPlayerRelativeBlockHardness(EntityPlayer ply, World world, int x, int y, int z) {
					return BlockMultipartBase.getPlayerRelativeBlockHardnessStatic(world, ply, x, y, z);
				} */
				
				MethodVisitor mv = super.visitMethod(Opcodes.ACC_PUBLIC, BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_NAME, BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_DESC, null, new String[0]);
				mv.visitCode();
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				mv.visitVarInsn(Opcodes.ALOAD, 2);
				mv.visitVarInsn(Opcodes.ILOAD, 3);
				mv.visitVarInsn(Opcodes.ILOAD, 4);
				mv.visitVarInsn(Opcodes.ILOAD, 5);
				mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "getPlayerRelativeBlockHardnessStatic", BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_DESC);
				mv.visitInsn(Opcodes.FRETURN);
				mv.visitMaxs(5, 6);
				mv.visitEnd();
			}
			
			/*{
				/* public ArrayList getBlockDropped(World, int, int, int, int, int) {
					return BlockMultipartBase.getBlockDroppedStatic();
				} * /
				
				MethodVisitor mv = super.visitMethod(Opcodes.ACC_PUBLIC, BLOCK_GETBLOCKDROPPED_NAME, BLOCK_GETBLOCKDROPPED_DESC, null, new String[0]);
				mv.visitCode();
				mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/core/multipart/BlockMultipartBase", "getBlockDroppedStatic", "()Ljava/util/ArrayList;");
				mv.visitInsn(Opcodes.ARETURN);
				mv.visitMaxs(1, 7);
				mv.visitEnd();
			} */
			
			{
				/* public MovingObjectPosition collisionRayTrace(World world, int x, int y, int z, Vec3 src, Vec3 dst) {
				 	return Util.collisionRayTrace(world, x, y, z, src, dst);
				} */
				
				MethodVisitor mv = super.visitMethod(Opcodes.ACC_PUBLIC, BLOCK_COLLISIONRAYTRACE_NAME, BLOCK_COLLISIONRAYTRACE_DESC, null, new String[0]);
				mv.visitCode();
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				mv.visitVarInsn(Opcodes.ILOAD, 2);
				mv.visitVarInsn(Opcodes.ILOAD, 3);
				mv.visitVarInsn(Opcodes.ILOAD, 4);
				mv.visitVarInsn(Opcodes.ALOAD, 5);
				mv.visitVarInsn(Opcodes.ALOAD, 6);
				mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "collisionRayTrace", BLOCK_COLLISIONRAYTRACE_DESC);
				mv.visitInsn(Opcodes.ARETURN);
				mv.visitMaxs(6, 7);
				mv.visitEnd();
			}
			
			super.visitEnd();
		}
	}
	
	public static final String IMCS_FIELD = "ImmibisCoreMicroblockCoverSystem";
	
	private static class TileTransformerVisitor extends ClassVisitorBase {
		public TileTransformerVisitor(ClassVisitor parent) {
			super(Opcodes.ASM4, parent);
		}
		
		private String position = "Centre";
		
		public String className, superclass;
		
		@Override
		public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
			interfaces = ObjectArrays.concat(interfaces, "immibis/core/api/microblock/IMicroblockSupporterTile");
			super.visit(version, access, name, signature, superName, interfaces);
			
			className = name;
			superclass = superName;
			
			FieldVisitor fv = super.visitField(Opcodes.ACC_PUBLIC, IMCS_FIELD, "Limmibis/core/api/microblock/IMicroblockCoverSystem;", null, null);
			fv.visitEnd();
			
			MethodVisitor mv = super.visitMethod(Opcodes.ACC_PUBLIC, "getPartPosition", "(I)Limmibis/core/api/microblock/EnumPosition;", null, new String[0]);
			mv.visitCode();
			mv.visitFieldInsn(Opcodes.GETSTATIC, "immibis/core/api/microblock/EnumPosition", position, "Limmibis/core/api/microblock/EnumPosition;");
			mv.visitInsn(Opcodes.ARETURN);
			mv.visitMaxs(1, 2);
			mv.visitEnd();
			
			{
				Label l = new Label();
				
				mv = super.visitMethod(Opcodes.ACC_PUBLIC, "isPlacementBlockedByTile", "(Limmibis/core/api/microblock/PartType;Limmibis/core/api/microblock/EnumPosition;)Z", null, new String[0]);
				mv.visitCode();
				mv.visitFieldInsn(Opcodes.GETSTATIC, "immibis/core/api/microblock/EnumPosition", position, "Limmibis/core/api/microblock/EnumPosition;");
				mv.visitVarInsn(Opcodes.ALOAD, 2);
				
				// return true if position argument = position of part
				mv.visitJumpInsn(Opcodes.IF_ACMPEQ, l);
				mv.visitInsn(Opcodes.ICONST_0);
				mv.visitInsn(Opcodes.IRETURN);
				
				mv.visitLabel(l);
				mv.visitFrame(Opcodes.F_SAME, 0, null, 0, null);
				mv.visitInsn(Opcodes.ICONST_1);
				mv.visitInsn(Opcodes.IRETURN);
				
				mv.visitMaxs(2, 3);
				mv.visitEnd();
			}
			
			{
				Label l = new Label();
				
				mv = super.visitMethod(Opcodes.ACC_PUBLIC, "isPositionOccupiedByTile", "(Limmibis/core/api/microblock/EnumPosition;)Z", null, new String[0]);
				mv.visitCode();
				mv.visitFieldInsn(Opcodes.GETSTATIC, "immibis/core/api/microblock/EnumPosition", position, "Limmibis/core/api/microblock/EnumPosition;");
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				
				// return true if position argument = position of part
				mv.visitJumpInsn(Opcodes.IF_ACMPEQ, l);
				mv.visitInsn(Opcodes.ICONST_0);
				mv.visitInsn(Opcodes.IRETURN);
				
				mv.visitLabel(l);
				mv.visitFrame(Opcodes.F_SAME, 0, null, 0, null);
				mv.visitInsn(Opcodes.ICONST_1);
				mv.visitInsn(Opcodes.IRETURN);
				
				mv.visitMaxs(2, 2);
				mv.visitEnd();
			}
			
			mv = super.visitMethod(Opcodes.ACC_PUBLIC, "getCoverSystem", "()Limmibis/core/api/microblock/IMicroblockCoverSystem;", null, new String[0]);
			mv.visitCode();
			mv.visitVarInsn(Opcodes.ALOAD, 0); // 'this'
			mv.visitFieldInsn(Opcodes.GETFIELD, className, IMCS_FIELD, "Limmibis/core/api/microblock/IMicroblockCoverSystem;");
			mv.visitInsn(Opcodes.ARETURN);
			mv.visitMaxs(1, 1);
			mv.visitEnd();
			
			mv = super.visitMethod(Opcodes.ACC_PUBLIC, "getCoverSystem", "()Limmibis/core/api/multipart/ICoverSystem;", null, new String[0]);
			mv.visitCode();
			mv.visitVarInsn(Opcodes.ALOAD, 0); // 'this'
			mv.visitFieldInsn(Opcodes.GETFIELD, className, IMCS_FIELD, "Limmibis/core/api/microblock/IMicroblockCoverSystem;");
			mv.visitInsn(Opcodes.ARETURN);
			mv.visitMaxs(1, 1);
			mv.visitEnd();
			
			mv = super.visitMethod(Opcodes.ACC_PUBLIC, "getPlayerRelativePartHardness", "(L"+ENTITYPLAYER+";I)F", null, new String[0]);
			mv.visitCode();
			mv.visitVarInsn(Opcodes.ALOAD, 0);
			mv.visitVarInsn(Opcodes.ALOAD, 1);
			mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "getPlayerRelativeTileHardness", "(L"+TILEENTITY+";L"+ENTITYPLAYER+";)F");
			mv.visitInsn(Opcodes.FRETURN);
			mv.visitMaxs(2, 3);
			mv.visitEnd();
			
			if(FMLRelauncher.side().equals("CLIENT")) {
				// public void renderPart(RenderBlocks rb, int) {render(rb);}
				mv = super.visitMethod(Opcodes.ACC_PUBLIC, "renderPart", "(L"+RENDERBLOCKS+";I)V", null, new String[0]);
				mv.visitCode();
				mv.visitVarInsn(Opcodes.ALOAD, 0);
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				mv.visitMethodInsn(Opcodes.INVOKEVIRTUAL, className, "render", "(L"+RENDERBLOCKS+";)V");
				mv.visitInsn(Opcodes.RETURN);
				mv.visitMaxs(2, 3);
				mv.visitEnd();
				
				// public void render(RenderBlocks rb) {Util.render(this, rb);}
				mv = super.visitMethod(Opcodes.ACC_PUBLIC, "render", "(L"+RENDERBLOCKS+";)V", null, new String[0]);
				mv.visitCode();
				mv.visitVarInsn(Opcodes.ALOAD, 0);
				mv.visitVarInsn(Opcodes.ALOAD, 1);
				mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "render", "(L"+TILEENTITY+";L"+RENDERBLOCKS+";)V");
				mv.visitInsn(Opcodes.RETURN);
				mv.visitMaxs(2, 2);
				mv.visitEnd();
			}
			
			// nothing calls getCollidingBoundingBoxes except BlockMultipartBase which isn't involved here, so we can leave it out
			// same for pickPart and isSolidOnSide
			
			mv = super.visitMethod(Opcodes.ACC_PUBLIC, "removePartByPlayer", "(L"+ENTITYPLAYER+";I)Ljava/util/List;", null, new String[0]);
			mv.visitCode();
			mv.visitVarInsn(Opcodes.ALOAD, 0);
			mv.visitVarInsn(Opcodes.ALOAD, 1);
			mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "removeTileByPlayer", "(L"+TILEENTITY+";L"+ENTITYPLAYER+";)Ljava/util/List;");
			mv.visitInsn(Opcodes.ARETURN);
			mv.visitMaxs(2, 3);
			mv.visitEnd();
			
			mv = super.visitMethod(Opcodes.ACC_PUBLIC, "getPartAABBFromPool", "(I)L"+AXISALIGNEDBB+";", null, new String[0]);
			mv.visitCode();
			mv.visitVarInsn(Opcodes.ALOAD, 0);
			mv.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "getTileAABBFromPool", "(L"+TILEENTITY+";)L"+AXISALIGNEDBB+";");
			mv.visitInsn(Opcodes.ARETURN);
			mv.visitMaxs(1, 2);
			mv.visitEnd();
		}
		
		private static class TransformMethodVisitor_NBT extends MethodVisitor {
			public String name;
			
			public TransformMethodVisitor_NBT(MethodVisitor parent, String name) {
				super(Opcodes.ASM4, parent);
				this.name = name;
			}
			
			@Override
			public void visitCode() {
				super.visitCode();
				
				visitVarInsn(Opcodes.ALOAD, 0); // this
				visitVarInsn(Opcodes.ALOAD, 1); // tag
				visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", name, "(L"+TILEENTITY+";L"+NBTTAGCOMPOUND+";)V");
			}
			
			@Override
			public void visitMaxs(int maxStack, int maxLocals) {
				if(maxStack < 2)
					maxStack = 2;
				
				super.visitMaxs(maxStack, maxLocals);
			}
		}
		
		private boolean saw_constructor;
		private boolean saw_getDescriptionPacket;
		private boolean saw_readFromNBT;
		private boolean saw_writeToNBT;
		
		@Override
		public MethodVisitor visitMethod(int access, String name, String desc, String signature, String[] exceptions) {
			MethodVisitor mv = super.visitMethod(access, name, desc, signature, exceptions);
			
			if(name.equals("<init>")) {
				// create IMicroblockCoverSystem in constructor
				mv = new MethodVisitor(Opcodes.ASM4, mv) {
					@Override
					public void visitInsn(int opcode) {
						if(opcode == Opcodes.RETURN) {
							super.visitVarInsn(Opcodes.ALOAD, 0);
							super.visitInsn(Opcodes.DUP);
							super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "createCoverSystem", "(Ljava/lang/Object;)Limmibis/core/api/microblock/IMicroblockCoverSystem;");
							super.visitFieldInsn(Opcodes.PUTFIELD, className, IMCS_FIELD, "Limmibis/core/api/microblock/IMicroblockCoverSystem;");
						}
						
						super.visitInsn(opcode);
					}
					
					@Override
					public void visitMaxs(int maxStack, int maxLocals) {
						if(maxStack < 2)
							maxStack = 2;
						
						super.visitMaxs(maxStack, maxLocals);
					}
				};
				saw_constructor = true;
			}
			
			if(name.equals(TILEENTITY_GETDESCRIPTIONPACKET_NAME) && desc.equals(TILEENTITY_GETDESCRIPTIONPACKET_DESC)) {
				saw_getDescriptionPacket = true;
				
				mv = new MethodVisitor(Opcodes.ASM4, mv) {
					@Override
					public void visitInsn(int opcode) {
						if(opcode == Opcodes.ARETURN) {
							super.visitVarInsn(Opcodes.ALOAD, 0);
							super.visitMethodInsn(Opcodes.INVOKESTATIC, "immibis/microblocks/coremod/MicroblockSupporterTransformer$Util", "getDescriptionPacket", "(L"+PACKET+";L"+TILEENTITY+";)L"+PACKET+";");
						}
						
						super.visitInsn(opcode);
					}
					
					@Override
					public void visitMaxs(int maxStack, int maxLocals) {
						if(maxStack < 2)
							maxStack = 2;
						
						super.visitMaxs(maxStack, maxLocals);
					}
				};
			}
			
			if(name.equals(TILEENTITY_READFROMNBT_NAME) && desc.equals(TILEENTITY_READFROMNBT_DESC)) {
				saw_readFromNBT = true;
				mv = new TransformMethodVisitor_NBT(mv, "readFromNBT");
			}
			
			if(name.equals(TILEENTITY_WRITETONBT_NAME) && desc.equals(TILEENTITY_WRITETONBT_DESC)) {
				saw_writeToNBT = true;
				mv = new TransformMethodVisitor_NBT(mv, "writeToNBT");
			}
			
			return mv;
		}
		
		@Override
		public void visitEnd() {
		
			if(!saw_constructor)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, "<init>", "()V");
			
			if(!saw_getDescriptionPacket)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, TILEENTITY_GETDESCRIPTIONPACKET_NAME, TILEENTITY_GETDESCRIPTIONPACKET_DESC);
			
			if(!saw_readFromNBT)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, TILEENTITY_READFROMNBT_NAME, TILEENTITY_READFROMNBT_DESC);
			
			if(!saw_writeToNBT)
				generateAndTransformMethod(this, superclass, Opcodes.ACC_PUBLIC, TILEENTITY_WRITETONBT_NAME, TILEENTITY_WRITETONBT_DESC);
			
			super.visitEnd();
		}
	}

	@Override
	public byte[] transform(String name, byte[] bytes) {
		if(blockClasses.contains(name)) {
			ClassWriter cw = new ClassWriter(0);
			new ClassReader(bytes).accept(new BlockTransformerVisitor(MCP ? new CheckClassAdapter(cw) : cw), 0);
			return cw.toByteArray();
		}
		
		if(tileClasses.contains(name)) {
			ClassWriter cw = new ClassWriter(0);
			new ClassReader(bytes).accept(new TileTransformerVisitor(MCP ? new CheckClassAdapter(cw) : cw), 0);
			return cw.toByteArray();
		}
		
		return bytes;
	}
	
	
	
	public static final String NBT_FIELD_NAME = "ImmibisCoreMicroblocks";
	
	
	// called by generated code
	public static class Util {
		public static IMicroblockCoverSystem createCoverSystem(Object tile) {
			IMicroblockSystem ims = APILocator.getMicroblockSystem();
			if(ims != null)
				return ims.createMicroblockCoverSystem((IMicroblockSupporterTile)tile);
			else
				return null;
		}
		
		public static float getPlayerRelativeTileHardness(TileEntity te, EntityPlayer ply) throws Exception {
			int blockID = te.worldObj.getBlockId(te.xCoord, te.yCoord, te.zCoord);
			Block block = Block.blocksList[blockID];
			
			Method m = block.getClass().getMethod(BLOCK_GETPLAYERRELATIVEBLOCKHARDNESS_REPLACEMENT, EntityPlayer.class, World.class, int.class, int.class, int.class);
			return (Float)m.invoke(block, ply, te.worldObj, te.xCoord, te.yCoord, te.zCoord);
		}
		
		@SideOnly(Side.CLIENT)
		public static void render(TileEntity te, RenderBlocks rb) {
			BlockMultipartBase.renderBlockStatic(rb, Block.blocksList[te.worldObj.getBlockId(te.xCoord, te.yCoord, te.zCoord)], te.xCoord, te.yCoord, te.zCoord);
		}

		
		// Marker itemstack. If the item dropped is this one, then instead of dropping it we do the normal block drops.  
		private static ItemStack dropNormalDrops = new ItemStack(0, 0, 0);

		public static List<ItemStack> removeTileByPlayer(TileEntity te, EntityPlayer ply) throws Exception {
			int blockID = te.worldObj.getBlockId(te.xCoord, te.yCoord, te.zCoord);
			Block block = Block.blocksList[blockID];
			
			//List<ItemStack> rv = (List<ItemStack>)block.getClass().getMethod(BLOCK_GETBLOCKDROPPED_REPLACEMENT, World.class, int.class, int.class, int.class, int.class, int.class)
			//		.invoke(block, te.worldObj, te.xCoord, te.yCoord, te.zCoord, te.worldObj.getBlockMetadata(te.xCoord, te.yCoord, te.zCoord), EnchantmentHelper.getFortuneModifier(ply));
			
			IMicroblockSupporterTile imst = (IMicroblockSupporterTile)te;
			
			if(!(Boolean)block.getClass()
				.getMethod(BLOCK_REMOVEBLOCKBYPLAYER_REPLACEMENT, World.class, EntityPlayer.class, int.class, int.class, int.class)
				.invoke(block, te.worldObj, ply, te.xCoord, te.yCoord, te.zCoord))
				return null;
			
			// if removeBlockByPlayer removed the block, then un-remove it and place the microblock container block
			if(te.worldObj.getBlockId(te.xCoord, te.yCoord, te.zCoord) == 0)
				imst.getCoverSystem().convertToContainerBlock();
			
			return Collections.singletonList(dropNormalDrops);
		}
		
		// return true to cancel normal block drops
		public static boolean dropBlockAsItemWithChance(World world, int x, int y, int z, float chance) {
			
			List<ItemStack> drops = BlockMultipartBase.getBlockDroppedStatic();
			if(drops == null || drops.size() == 0)
				return true;
			
			if(drops.size() == 1 && drops.get(0) == dropNormalDrops)
				// player broke the actual block, do normal block dropping
				return false;
			
			// do microblock drops

            for (ItemStack item : drops)
            {
                if (world.rand.nextFloat() <= chance)
                {
                	if (!world.isRemote && world.getGameRules().getGameRuleBooleanValue("doTileDrops"))
                    {
                        float var6 = 0.7F;
                        double var7 = (double)(world.rand.nextFloat() * var6) + (double)(1.0F - var6) * 0.5D;
                        double var9 = (double)(world.rand.nextFloat() * var6) + (double)(1.0F - var6) * 0.5D;
                        double var11 = (double)(world.rand.nextFloat() * var6) + (double)(1.0F - var6) * 0.5D;
                        EntityItem var13 = new EntityItem(world, (double)x + var7, (double)y + var9, (double)z + var11, item);
                        var13.delayBeforeCanPickup = 10;
                        world.spawnEntityInWorld(var13);
                    }
                }
            }
			
			return true;
		}
		
		public static AxisAlignedBB getTileAABBFromPool(TileEntity te) {
			int blockID = te.worldObj.getBlockId(te.xCoord, te.yCoord, te.zCoord);
			Block block = Block.blocksList[blockID];
			
			AxisAlignedBB bb;
			if(te.worldObj.isRemote) {
				bb = block.getSelectedBoundingBoxFromPool(te.worldObj, te.xCoord, te.yCoord, te.zCoord);
			} else {
				bb = block.getCollisionBoundingBoxFromPool(te.worldObj, te.xCoord, te.yCoord, te.zCoord);
			}
			if(bb == null)
				return AxisAlignedBB.getAABBPool().addOrModifyAABBInPool(0, 0, 0, 1, 1, 1);
			else
				return bb.offset(-te.xCoord, -te.yCoord, -te.zCoord);
		}
		
		public static Packet getDescriptionPacket(Packet originalPacket, TileEntity te) {
			IMicroblockCoverSystem mcs = ((IMicroblockSupporterTile)te).getCoverSystem();
			PacketMicroblockContainerDescription pmcd;
			
			if(originalPacket != null) {
				pmcd = new PacketMicroblockDescriptionWithWrapping();
				((PacketMicroblockDescriptionWithWrapping)pmcd).wrappedPacket = originalPacket;
			} else {
				pmcd = new PacketMicroblockContainerDescription();
			}
			
			pmcd.x = te.xCoord;
			pmcd.y = te.yCoord;
			pmcd.z = te.zCoord;
			pmcd.data = mcs.writeDescriptionBytes();
			
			Packet wrapped = APILocator.getNetManager().wrap(pmcd);
			wrapped.isChunkDataPacket = true;
			return wrapped;
		}
		
		public static void writeToNBT(TileEntity te, NBTTagCompound tag) {
			IMicroblockCoverSystem imcs = ((IMicroblockSupporterTile)te).getCoverSystem();
			
			NBTTagCompound csTag = new NBTTagCompound();
			imcs.writeToNBT(csTag);
			tag.setTag(NBT_FIELD_NAME, csTag);
		}
		
		public static void readFromNBT(TileEntity te, NBTTagCompound tag) {
			IMicroblockCoverSystem imcs = ((IMicroblockSupporterTile)te).getCoverSystem();
			
			if(tag.hasKey(NBT_FIELD_NAME))
				imcs.readFromNBT(tag.getCompoundTag(NBT_FIELD_NAME));
		}
		
		public static MovingObjectPosition collisionRayTrace(World world, int x, int y, int z, Vec3 src, Vec3 dst) throws Throwable {
			IMultipartTile tile = (IMultipartTile)world.getBlockTileEntity(x, y, z);
			
			ICoverSystem ci = tile.getCoverSystem();
			MovingObjectPosition ciPos = ci == null ? null : ci.collisionRayTrace(src, dst);
			
			int blockID = world.getBlockId(x, y, z);
			Block block = Block.blocksList[blockID];
			
			MovingObjectPosition tilePos = (MovingObjectPosition)block.getClass()
					.getMethod(BLOCK_COLLISIONRAYTRACE_REPLACEMENT, World.class, int.class, int.class, int.class, Vec3.class, Vec3.class)
					.invoke(block, world, x, y, z, src, dst);
			
			if(tilePos != null && tilePos.subHit != -1)
				throw new RuntimeException("MicroblockSupporterTransformer cannot be used on blocks with multiple selection boxes. Offending block was "+block+" at ID "+block.blockID+". subHit value was "+tilePos.subHit);
			
			if(tilePos != null)
				tilePos.subHit = 0;
			
			if(tilePos == null) return ciPos;
			if(ciPos == null) return tilePos;
			
			double ciDist = ciPos.hitVec.squareDistanceTo(src);
			double tileDist = tilePos.hitVec.squareDistanceTo(src);
			
			return ciDist < tileDist ? ciPos : tilePos;
		}
		
		public static boolean isSolidOnSide(World world, int x, int y, int z, ForgeDirection side) {
			IMultipartTile tile = (IMultipartTile)world.getBlockTileEntity(x, y, z);
			return tile.getCoverSystem().isSolidOnSide(side);
		}
		
		public static ItemStack pickPart(MovingObjectPosition rayTrace, World w, int x, int y, int z) {
			if(rayTrace.subHit >= 0)
				return null;
			IMultipartTile tile = (IMultipartTile)w.getBlockTileEntity(x, y, z);
			return tile.getCoverSystem().pickPart(rayTrace, -1 - rayTrace.subHit);
		}
	}
}
