package immibis.ccperiphs.forth;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Scanner;
import java.util.Stack;
import java.util.Map.Entry;
import net.minecraft.server.NBTTagCompound;
import net.minecraft.server.NBTTagInt;
import net.minecraft.server.NBTTagList;
import net.minecraft.server.NBTTagShort;

public class ForthContext
{
    private static final int INITIAL_CODESIZE = 64;
    private static final int MAX_CODESIZE = 32767;
    private final int INSTRS_PER_TICK;
    private final int MAX_SAVED_INSTRS;
    Scanner tib = null;
    public IOutputDevice out;
    private short[] code;
    private short codesize;
    JavaDictionary java_dict;
    Map forth_dict;
    private Stack stack = new Stack();
    private int[] rstack = new int[64];
    private int rstack_pos;
    private boolean halted = false;
    private boolean fatal = false;
    private String defining = null;
    private Queue immed = new LinkedList();
    private int instrs;
    private boolean waiting;
    private static JavaDictionary BASE_DICT = new ForthBaseDictionary();

    public void error(String var1)
    {
        if (!this.halted)
        {
            this.out.write(var1 + "\n");
        }

        this.halted = true;
    }

    public void doReturn()
    {
        this.rpop();
    }

    public int rpop()
    {
        if (this.rstack_pos == -1)
        {
            this.error("Return stack empty");
            return 0;
        }
        else
        {
            return this.rstack[this.rstack_pos--];
        }
    }

    public int rpick(int var1)
    {
        if (this.rstack_pos < var1)
        {
            this.error("Return stack empty");
            return 0;
        }
        else
        {
            return this.rstack[this.rstack_pos - var1];
        }
    }

    public void rset(int var1, int var2)
    {
        if (this.rstack_pos < var1)
        {
            this.error("Return stack empty");
        }
        else
        {
            this.rstack[this.rstack_pos - var1] = var2;
        }
    }

    public int popInt()
    {
        Object var1 = this.pop();

        if (var1 instanceof Number)
        {
            return ((Number)var1).intValue();
        }
        else
        {
            this.error("Expected number, got " + var1);
            return 0;
        }
    }

    public Number popNumber()
    {
        Object var1 = this.pop();

        if (var1 instanceof Number)
        {
            return (Number)var1;
        }
        else
        {
            this.error("Expected number, got " + var1);
            return null;
        }
    }

    public Scanner getTIB()
    {
        return this.tib;
    }

    public short nextCode()
    {
        if (this.rstack_pos == -1)
        {
            if (this.immed.isEmpty())
            {
                this.error("nextCode() called outside of word");
                return (short)0;
            }
            else
            {
                return ((Short)this.immed.poll()).shortValue();
            }
        }
        else if (this.rstack[this.rstack_pos] >= this.codesize)
        {
            this.error("End of code reached (at " + this.rstack[this.rstack_pos] + ")");
            this.halted = true;
            return (short)0;
        }
        else
        {
            return this.code[this.rstack[this.rstack_pos]++];
        }
    }

    private void disassemble()
    {
        HashMap var1 = new HashMap();
        Iterator var3 = BASE_DICT.getAllWords().iterator();
        String var2;

        while (var3.hasNext())
        {
            var2 = (String)var3.next();
            var1.put(Integer.valueOf(49152 | BASE_DICT.getId(var2)), var2);
        }

        var3 = this.java_dict.getAllWords().iterator();

        while (var3.hasNext())
        {
            var2 = (String)var3.next();
            var1.put(Integer.valueOf(32768 | this.java_dict.getId(var2)), var2);
        }

        var3 = this.forth_dict.keySet().iterator();

        while (var3.hasNext())
        {
            var2 = (String)var3.next();
            var1.put(Integer.valueOf(((Short)this.forth_dict.get(var2)).shortValue()), var2);
        }

        for (int var4 = 0; var4 < this.codesize; ++var4)
        {
            int var5 = this.code[var4] & 65535;

            if (var1.containsKey(Integer.valueOf(var5)))
            {
                System.out.println(var4 + ": " + var5 + " (" + (String)var1.get(Integer.valueOf(var5)) + ")");
            }
            else
            {
                System.out.println(var4 + ": " + var5);
            }
        }
    }

    public void doCall(short var1)
    {
        this.rpush(var1);
    }

    public void rpush(int var1)
    {
        if (this.rstack_pos == this.rstack.length - 1)
        {
            this.error("Return stack overflow");
            this.halted = true;
        }
        else
        {
            this.rstack[++this.rstack_pos] = var1;
        }
    }

    public Object pop()
    {
        if (this.stack.isEmpty())
        {
            this.error("Empty stack");
            this.halted = true;
            return null;
        }
        else
        {
            return this.stack.pop();
        }
    }

    public void push(Object var1)
    {
        if (this.stack.size() > 256)
        {
            this.error("Stack overflow");
            this.halted = true;
        }
        else
        {
            this.stack.push(var1);
        }
    }

    public void write(String var1)
    {
        this.out.write(var1);
    }

    public void writeWrapped(String var1)
    {
        for (boolean var10000 = true; var1.length() > 50; var1 = var1.substring(50))
        {
            this.write(var1.substring(0, 50) + "\n");
        }

        if (var1.length() > 0)
        {
            this.write(var1 + "\n");
        }
    }

    public ForthContext(IOutputDevice var1, JavaDictionary var2, int var3, int var4)
    {
        this.out = var1;
        this.java_dict = var2;
        this.INSTRS_PER_TICK = var3;
        this.MAX_SAVED_INSTRS = var4;
        this.reset();
    }

    public String readWord()
    {
        return this.tib.hasNext() ? this.tib.next() : "";
    }

    public void appendCode(short var1)
    {
        if (this.defining != null)
        {
            if (this.codesize == this.code.length - 1)
            {
                if (this.codesize == 32766)
                {
                    this.out.write("Out of memory\n");
                    this.halted = true;
                    this.fatal = true;
                    return;
                }

                short[] var2 = new short[this.code.length * 2];
                System.arraycopy(this.code, 0, var2, 0, this.code.length);
                this.code = var2;
            }

            this.code[this.codesize++] = var1;
        }
        else
        {
            this.immed.add(Short.valueOf(var1));
        }
    }

    public void compileWord(String var1)
    {
        if (this.forth_dict.containsKey(var1))
        {
            this.appendCode(((Short)this.forth_dict.get(var1)).shortValue());
        }
        else
        {
            short var2;

            if (this.java_dict.getId(var1) > 0)
            {
                var2 = this.java_dict.getId(var1);
                this.appendCode((short)(var2 | 32768));
            }
            else if (BASE_DICT.getId(var1) > 0)
            {
                var2 = BASE_DICT.getId(var1);
                this.appendCode((short)(var2 | 49152));
            }
            else
            {
                this.error("Unknown word: " + var1);
            }
        }
    }

    public short HERE()
    {
        return this.codesize;
    }

    public void writeCode(int var1, short var2)
    {
        if (var1 < this.codesize && var1 >= 0)
        {
            this.code[var1] = var2;
        }
    }

    public void interpret(Scanner var1)
    {
        this.tib = var1;
        this.defining = null;
        this.immed.clear();

        while (var1.hasNext() && !this.fatal)
        {
            String var2 = this.readWord();

            if (this.forth_dict.containsKey(var2))
            {
                this.appendCode(((Short)this.forth_dict.get(var2)).shortValue());
            }
            else
            {
                short var3;
                IJavaWord var4;

                if (this.java_dict.getId(var2) > 0)
                {
                    var3 = this.java_dict.getId(var2);
                    var4 = this.java_dict.getWord(var3);

                    if (var4 != null)
                    {
                        if (var4 instanceof IImmediateWord)
                        {
                            var4.execute(this);
                        }
                        else
                        {
                            this.appendCode((short)(var3 | 32768));
                        }
                    }
                    else
                    {
                        (new Exception("java_dict.getWord(" + var3 + ") is null even though java_dict.getId(\"" + var2 + "\") == " + var3)).printStackTrace();
                    }
                }
                else if (BASE_DICT.getId(var2) > 0)
                {
                    var3 = BASE_DICT.getId(var2);
                    var4 = BASE_DICT.getWord(var3);

                    if (var4 != null)
                    {
                        if (var4 instanceof IImmediateWord)
                        {
                            var4.execute(this);
                        }
                        else
                        {
                            this.appendCode((short)(var3 | 49152));
                        }
                    }
                    else
                    {
                        (new Exception("BASE_DICT.getWord(" + var3 + ") is null even though BASE_DICT.getId(\"" + var2 + "\") == " + var3)).printStackTrace();
                    }
                }
                else if (var2.equals(":"))
                {
                    String var6 = this.readWord();

                    if (this.defining != null)
                    {
                        this.out.write(": found while compiling (" + var6 + " is inside " + this.defining + ")\n");
                        this.halted = true;
                        this.fatal = true;
                        return;
                    }

                    this.forth_dict.put(var6, Short.valueOf(this.codesize));
                    this.defining = var6;
                }
                else if (var2.equals(";"))
                {
                    if (this.defining == null)
                    {
                        this.out.write("; found while not compiling a word\n");
                        this.halted = true;
                        this.fatal = true;
                        return;
                    }

                    this.appendCode((short)(BASE_DICT.getId("RETURN") | 49152));
                    this.defining = null;
                }
                else
                {
                    try
                    {
                        int var7 = Integer.parseInt(var2);

                        if ((var7 & -65536) == 0)
                        {
                            this.appendCode((short)(BASE_DICT.getId("(lit)") | 49152));
                            this.appendCode((short)var7);
                        }
                        else
                        {
                            this.appendCode((short)(BASE_DICT.getId("(lit2)") | 49152));
                            this.appendCode((short)var7);
                            this.appendCode((short)(var7 >> 16));
                        }
                    }
                    catch (NumberFormatException var5)
                    {
                        this.out.write("Unknown word: " + var2 + "\n");
                        this.halted = true;
                    }
                }
            }
        }

        if (this.defining != null)
        {
            this.out.write("Unclosed definition (of " + this.defining + ")");
            this.halted = true;
        }

        this.defining = null;
    }

    public void waitForNextTick()
    {
        this.waiting = true;
    }

    public void tick()
    {
        this.waiting = false;

        for (this.instrs = Math.min(this.instrs + this.INSTRS_PER_TICK, this.MAX_SAVED_INSTRS); !this.waiting && !this.halted && this.instrs > 0; --this.instrs)
        {
            this.runInstr();
        }
    }

    public void runInstr()
    {
        short var1;

        if (this.rstack_pos != -1)
        {
            var1 = this.code[this.rstack[this.rstack_pos]++];
        }
        else
        {
            if (this.immed.isEmpty())
            {
                this.waiting = true;
                return;
            }

            var1 = ((Short)this.immed.poll()).shortValue();
        }

        if (var1 < 0)
        {
            IJavaWord var2 = ((var1 & 49152) == 49152 ? BASE_DICT : this.java_dict).getWord((short)(var1 & 16383));

            if (var2 == null)
            {
                this.error("Invalid word ID: " + var1);
                this.halted = true;
                return;
            }

            var2.execute(this);
        }
        else
        {
            this.doCall(var1);
        }
    }

    public void halt()
    {
        this.halted = true;
    }

    public NBTTagCompound save()
    {
        NBTTagCompound var1 = new NBTTagCompound();
        var1.setShort("codesize", this.codesize);
        var1.setInt("instrs", this.instrs);
        var1.setBoolean("halted", this.halted);
        var1.setBoolean("fatal", this.fatal);
        NBTTagList var2 = new NBTTagList();
        Iterator var4 = this.immed.iterator();

        while (var4.hasNext())
        {
            short var3 = ((Short)var4.next()).shortValue();
            var2.add(new NBTTagShort("", var3));
        }

        var1.set("immed", var2);
        var2 = new NBTTagList();

        for (int var6 = 0; var6 <= this.rstack_pos; ++var6)
        {
            var2.add(new NBTTagInt("", this.rstack[var6]));
        }

        var1.set("rstack", var2);
        var2 = new NBTTagList();
        var4 = this.stack.iterator();

        while (var4.hasNext())
        {
            Integer var7 = (Integer)var4.next();
            var2.add(new NBTTagInt("", var7.intValue()));
        }

        var1.set("stack", var2);
        var2 = new NBTTagList();
        var4 = this.forth_dict.entrySet().iterator();

        while (var4.hasNext())
        {
            Entry var8 = (Entry)var4.next();
            NBTTagCompound var5 = new NBTTagCompound();
            var5.setString("k", (String)var8.getKey());
            var5.setShort("v", ((Short)var8.getValue()).shortValue());
            var2.add(var5);
        }

        var1.set("forth_dict", var2);
        byte[] var9 = new byte[this.codesize * 2];

        for (int var10 = 0; var10 < this.codesize; ++var10)
        {
            var9[var10 * 2] = (byte)this.code[var10];
            var9[var10 * 2 + 1] = (byte)(this.code[var10] >> 8);
        }

        var1.setByteArray("code", var9);
        return var1;
    }

    public void load(NBTTagCompound var1)
    {
        this.codesize = var1.getShort("codesize");
        this.instrs = var1.getInt("instrs");
        this.halted = var1.getBoolean("halted");
        this.fatal = var1.getBoolean("fatal");
        NBTTagList var2 = var1.getList("immed");
        this.immed.clear();
        int var3;

        for (var3 = 0; var3 < var2.size(); ++var3)
        {
            this.immed.add(Short.valueOf(((NBTTagShort)var2.get(var3)).data));
        }

        var2 = var1.getList("rstack");
        this.rstack_pos = Math.min(var2.size() - 1, this.rstack.length - 1);

        for (var3 = 0; var3 <= this.rstack_pos; ++var3)
        {
            this.rstack[var3] = ((NBTTagInt)var2.get(var3)).data;
        }

        this.stack.clear();
        var2 = var1.getList("stack");

        for (var3 = 0; var3 < var2.size(); ++var3)
        {
            this.stack.push(Integer.valueOf(((NBTTagInt)var2.get(var3)).data));
        }

        this.forth_dict.clear();
        var2 = var1.getList("forth_dict");

        for (var3 = 0; var3 < var2.size(); ++var3)
        {
            NBTTagCompound var4 = (NBTTagCompound)var2.get(var3);
            this.forth_dict.put(var4.getString("k"), Short.valueOf(var4.getShort("v")));
        }

        this.code = new short[this.codesize];
        byte[] var6 = var1.getByteArray("code");

        for (int var5 = 0; var5 < var6.length - 1 && var5 < this.codesize * 2; var5 += 2)
        {
            this.code[var5 / 2] = (short)(var6[var5] & 255 | var6[var5 + 1] << 8);
        }
    }

    public void reset()
    {
        this.code = new short[64];
        this.codesize = 0;
        this.forth_dict = new HashMap();
        this.rstack_pos = -1;
        this.halted = false;
        this.fatal = false;
        this.instrs = 0;
        this.waiting = false;
        this.stack.clear();
    }

    public void reboot()
    {
        this.rstack_pos = -1;
        this.halted = false;
        this.fatal = false;
        this.instrs = 0;
        this.waiting = false;
        this.stack.clear();
    }

    public int getForthWord(String var1)
    {
        return this.forth_dict.containsKey(var1) ? ((Short)this.forth_dict.get(var1)).shortValue() : -1;
    }

    public void queue(int var1)
    {
        if (this.immed.size() < 50)
        {
            this.immed.add(Short.valueOf((short)var1));
        }
    }

    public void jump(short var1)
    {
        this.rpop();
        this.rpush(var1);
    }

    public void createWordHere(String var1)
    {
        this.forth_dict.put(var1, Short.valueOf(this.codesize));
        this.defining = var1;
    }

    public short readCode(int var1)
    {
        return var1 < this.codesize && var1 >= 0 ? this.code[var1] : 0;
    }

    public void endDefinition()
    {
        this.defining = null;
    }

    public void compileString(String var1)
    {
        for (int var2 = 0; var2 < var1.length(); ++var2)
        {
            this.appendCode((short)var1.charAt(var2));
        }

        this.appendCode((short)0);
    }

    public String readCompiledString()
    {
        StringBuffer var1 = new StringBuffer();

        while (this.instrs > -100)
        {
            char var2 = (char)this.nextCode();

            if (var2 == 0)
            {
                break;
            }

            --this.instrs;
            var1.append(var2);
        }

        if (this.instrs <= -100)
        {
            this.error("compiled string too long");
        }

        return var1.toString();
    }

    public void setMemoryContents(short[] var1)
    {
        if (var1.length > 32768)
        {
            throw new IllegalArgumentException("length must be 32768 or less");
        }
        else
        {
            this.code = var1;
            this.codesize = var1.length > 32767 ? 32767 : (short)var1.length;
        }
    }
}
