package mods.immibis.infiview;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.DoubleBuffer;
import org.lwjgl.opengl.GL11;
import org.lwjgl.util.vector.Vector3f;

/*
 * Direction allocation:
 * 
 * From below:
 * 1 from 90 degrees (0)
 * 8 from 30 degrees (1-8)
 * 
 * From above:
 * 1 from 90 degrees (9)
 * 8 from 60 degrees (10-17)
 * 8 from 30 degrees (18-25)
 * 16 from 10 degrees (26-41)
 */
public class InfiViewDirections {
	
	public static final int NUM_DIRECTIONS = 42;
	
	// Returns the number of blocks wide between the centre of an image and its border.
	// This can be bigger than 8 to allow for diagonal images.
	public static int getImageDimensionIngame(int direction) {
		return 12; // TODO decide and make based on direction?
	}
	
	public static final Vector3f[] normalByDirection = new Vector3f[NUM_DIRECTIONS];
	public static final Vector3f[] udirByDirection = new Vector3f[NUM_DIRECTIONS]; // World direction corresponding to X direction on texture
	public static final Vector3f[] vdirByDirection = new Vector3f[NUM_DIRECTIONS]; // World direction corresponding to Y direction on texture
	
	static {
		double sin10 = Math.sin(Math.toRadians(10));
		double cos10 = Math.cos(Math.toRadians(10));
		double sin30 = Math.sin(Math.toRadians(30));
		double cos30 = Math.cos(Math.toRadians(30));
		double sin60 = Math.sin(Math.toRadians(60));
		double cos60 = Math.cos(Math.toRadians(60));
		double ang45 = Math.toRadians(45);
		double ang22p5 = Math.toRadians(22.5);
		
		normalByDirection[0] = new Vector3f(0, -1, 0);
		for(int k = 0; k < 8; k++)
			normalByDirection[k+1] = new Vector3f((float)(Math.sin(k*ang45)*cos30), -(float)sin30, (float)(Math.cos(k*ang45)*cos30));
		normalByDirection[9] = new Vector3f(0, 1, 0);
		for(int k = 0; k < 8; k++)
			normalByDirection[k+10] = new Vector3f((float)(Math.sin(k*ang45)*cos60), (float)sin60, (float)(Math.cos(k*ang45)*cos60));
		for(int k = 0; k < 8; k++)
			normalByDirection[k+18] = new Vector3f((float)(Math.sin(k*ang45)*cos30), (float)sin30, (float)(Math.cos(k*ang45)*cos30));
		for(int k = 0; k < 16; k++)
			normalByDirection[k+26] = new Vector3f((float)(Math.sin(k*ang22p5)*cos10), (float)sin10, (float)(Math.cos(k*ang22p5)*cos10));
		
		udirByDirection[0] = new Vector3f(1, 0, 0);
		vdirByDirection[0] = new Vector3f(0, 0, 1);
		udirByDirection[9] = new Vector3f(1, 0, 0);
		vdirByDirection[9] = new Vector3f(0, 0, 1);
		
		Vector3f up = new Vector3f(0, 1, 0);
		for(int k = 0; k < NUM_DIRECTIONS; k++) {
			if(k == 0 || k == 9)
				continue;
			Vector3f normal = normalByDirection[k];
			Vector3f udir = (Vector3f)Vector3f.cross(normal, up, new Vector3f()).normalise();
			Vector3f vdir = (Vector3f)Vector3f.cross(normal, udir, new Vector3f()).normalise();
			udirByDirection[k] = udir;
			vdirByDirection[k] = vdir;
		}
		
		for(int k = 0; k < NUM_DIRECTIONS; k++)
			if(normalByDirection[k] == null || udirByDirection[k] == null || vdirByDirection[k] == null)
				throw new AssertionError();
	}
	
	public static int getClosestDirection(float dx, float dy, float dz) {
		// avoid division by zero
		if(dx == 0) dx = 0.0001f;
		if(dz == 0) dz = 0.0001f;
		
		double fpitch = -Math.toDegrees(Math.atan(dy / Math.sqrt(dx*dx + dz*dz))); // range -90 to 90
		double fyaw = Math.toDegrees(Math.atan2(dx, dz)) + 180;
		if(fyaw < 0) fyaw += 360;
		if(fyaw >= 360) fyaw -= 360;
		// fyaw has range 0 to 360
		
		if(fpitch < -20)
		{
			if(fpitch < -45)
				return 0; // bottom view
			return 1 + ((int)(fyaw / 45 + 0.5f) & 7);
		}
		if(fpitch < 25)
			return 26 + ((int)(fyaw / 22.5f + 0.5f) & 7); // +10 degree view
		if(fpitch < 45)
			return 18 + ((int)(fyaw / 45 + 0.5f) & 7); // +30 degree view
		if(fpitch < 75)
			return 10 + ((int)(fyaw / 45 + 0.5f) & 15); // +60 degree view
		return 9; // top view
	}
	
	private static DoubleBuffer matrix = ByteBuffer.allocateDirect(8*16).order(ByteOrder.nativeOrder()).asDoubleBuffer();
	public static void setGLViewAngleByDirection(int direction) {
		Vector3f xdir = udirByDirection[direction];
		Vector3f ydir = vdirByDirection[direction];
		Vector3f zdir = normalByDirection[direction]; // this is actually the opposite of the z direction, so we negate it below
		
		matrix.position(0);
		matrix.put(xdir.x);
		matrix.put(ydir.x);
		matrix.put(-zdir.x);
		matrix.put(0);
		matrix.put(xdir.y);
		matrix.put(ydir.y);
		matrix.put(-zdir.y);
		matrix.put(0);
		matrix.put(xdir.z);
		matrix.put(ydir.z);
		matrix.put(-zdir.z);
		matrix.put(0);
		matrix.put(0);
		matrix.put(0);
		matrix.put(0);
		matrix.put(1);
		matrix.position(0);
		GL11.glMultMatrix(matrix);
	}
	
	public static void getDisplayDir(int direction, Vector3f udir, Vector3f vdir) {
		udir.set(udirByDirection[direction]);
		vdir.set(vdirByDirection[direction]);
	}
	
	
}
