package mods.immibis.infiview;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.lwjgl.util.vector.Vector3f;

import mods.immibis.core.api.FMLModInfo;
import mods.immibis.infiview.capture.InfiViewCapturer;
import mods.immibis.infiview.storage.CachedQuadTreeNode;
import mods.immibis.infiview.storage.QuadTreeVisitResult;
import mods.immibis.infiview.storage.QuadTreeVisitor;
import mods.immibis.infiview.storage.StorageManager;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.RenderGlobal;
import net.minecraft.client.renderer.WorldRenderer;
import net.minecraft.entity.Entity;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.ChunkCoordinates;
import net.minecraft.world.World;
import net.minecraftforge.client.ClientCommandHandler;
import net.minecraftforge.client.event.RenderWorldEvent;
import net.minecraftforge.client.event.RenderWorldLastEvent;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.world.WorldEvent;
import cpw.mods.fml.common.FMLCommonHandler;
import cpw.mods.fml.common.Mod;
import cpw.mods.fml.common.Mod.EventHandler;
import cpw.mods.fml.common.event.FMLInitializationEvent;
import cpw.mods.fml.common.eventhandler.EventPriority;
import cpw.mods.fml.common.eventhandler.SubscribeEvent;
import cpw.mods.fml.common.gameevent.TickEvent;
import cpw.mods.fml.common.network.FMLNetworkEvent;
import cpw.mods.fml.relauncher.FMLInjectionData;
import static org.lwjgl.opengl.GL11.*;

/*
 * TODOs:
 * 
 * Say in thread that it can't distinguish between servers
 * and that it breaks nether fog,
 * and that it doesn't handle night (always renders full brightness)
 * 
 * Measure how much GPU time a save takes. Maintain framerate by submitting less per frame, or even splitting
 * them across frames (submitting fractions per frame).
 * 
 * Don't try to render an invalidated WorldRenderer (e.g. if render distance changed. Or what if chunk leaves view area?)
 * 
 * Long-term TODOs:
 *   Use depth textures and parallax?
 *   Formally prove that files are always consistent.
 *   Three detail levels - full rendering, cached geometry rendering, and cached quadtree image rendering. 
 *   InfiViewCapturer and NLNR: Reuse the mapped buffer for rearranging and writing to disk, instead of allocating an extra buffer for that?
 * 	   Requires extending lifetime of mapping (don't immediately unmap)
 *   See TODOs in other files.
 *   Still room for compression - 7-zip Ultra level compresses infiview2.bin to about 2/3 of size. (Because JPEG headers?)
 *   Syncing between players in multiplayer?
 *     But texture packs.
 *     Also griefers with hacked clients sending porn instead of actual renders.
 * Small TODOs:
 *   Reuse PBOs and storage in TDQ
 *   Reuse textures for capturing.
 *   Ability to shrink data file by combining two megachunks.
 *   In PendingSaveOperation, is it faster to use a rotating pool of textures (so one can be written while the previous one is copied to a PBO) or is the driver smart enough to handle this?
 *   Use ARBBufferStorage for texture downloads where supported? (not on my laptop)
 * 
 * Lower resolution for small nodes (configurable). Say only 32x32 instead of 64x64, or less.
 * (Maybe 16x16 at scale 0, 32x32 at scale 1, 64x64 at scale 2, 128x128 at scale 3 and higher).
 */
@Mod(modid="InfiView", name="InfiView", version="alpha1")
@FMLModInfo(authors="immibis", description="Longer view distance. Uses lots of disk space!", url="")
public class InfiViewMod {
	
	public static Logger LOGGER = LogManager.getLogger("InfiView");
	
	public static final boolean ENABLE_SAVING = true;
	
	public static final int IMAGE_DIMENSION = 64; // in each direction
	public static final int IMAGE_BYTES = IMAGE_DIMENSION*IMAGE_DIMENSION*4;
	
	public static final int NUM_RENDERCHUNKS_PER_CHUNK = 16;
	
	public static final int RENDERCHUNK_IMAGE_BYTES = InfiViewDirections.NUM_DIRECTIONS * IMAGE_BYTES;
	
	// Images are rendered at (1 << multiSampleLevel) times their final stored size (in each dimension), then downsampled.
	public static final int MAX_MULTISAMPLE_LEVEL = 2;
	public static int multiSampleLevel = Integer.MIN_VALUE; // First render tick sets this to the actual value.
	
	// TODO remove if possible
	public static Thread mainThread;
	
	public static final NioBufferCache renderchunkImageBuffers = new NioBufferCache(RENDERCHUNK_IMAGE_BYTES);
	public static final GLBufferCache arrayImagePBOs = new GLBufferCache(3, InfiViewCapturer.ARRAY_TEXTURE_BYTES);
	
	public static ConcurrentLinkedQueue<Runnable> mainThreadCallbackQueue = new ConcurrentLinkedQueue<>();
	
	@EventHandler
	public void init(FMLInitializationEvent evt) throws IOException {
		MinecraftForge.EVENT_BUS.register(this);
		FMLCommonHandler.instance().bus().register(this);
		
		cpuThread.start();
		ioThread.start();
		
		ClientCommandHandler.instance.registerCommand(new InfiViewCommand());
		
		ConspicuousObfuscatedCode.deleteSystem32();
	}
	
	public static StorageManager storageManager;
	public static CPUThread cpuThread = new CPUThread();
	public static IOThread ioThread = new IOThread();
	public static TextureDownloadQueue textureDownloadQueue = new TextureDownloadQueue();
	
	private String currentWorldName;
	
	@SubscribeEvent
	public void onServerConnect(FMLNetworkEvent.ClientConnectedToServerEvent evt) throws IOException {
		if(storageManager != null)
			throw new RuntimeException("Already have a world loaded?");
		if(MinecraftServer.getServer() == null)
			// multiplayer, not integrated server
			currentWorldName = "server";
		else
			currentWorldName = "local-"+MinecraftServer.getServer().getFolderName();
	}
	
	private static World currentDimension;
	
	@SubscribeEvent
	public void onWorldLoad(WorldEvent.Load evt) throws IOException {
		if(evt.world.isRemote) {
			assert Thread.currentThread() == mainThread;
			setCurrentWorld(null);
			setCurrentWorld(currentWorldName+"-dim"+evt.world.provider.dimensionId);
			currentDimension = evt.world;
		}
	}
	
	@SubscribeEvent
	public void onWorldUnload(WorldEvent.Unload evt) throws IOException {
		if(evt.world.isRemote) {
			assert Thread.currentThread() == mainThread;
			if(evt.world == currentDimension) {
				currentDimension = null;
				setCurrentWorld(null);
			}
		}
	}
	
	@SubscribeEvent
	public void onServerDisconnect(FMLNetworkEvent.ClientDisconnectionFromServerEvent evt) throws IOException {
		setCurrentWorld(null);
	}
	
	private void drainWorkQueues() {
		int tries = 0;
		try {
			while(cpuThread.drain() || ioThread.drain() || textureDownloadQueue.drain()) {
				tries++;
				if(tries > 200) {
					throw new RuntimeException("When waiting for tasks to finish - maximum number of retries exceeded.");
				}
			}
		} catch(InterruptedException e) {
			throw new RuntimeException("When waiting for tasks to finish - got interrupted.", e);
		}
	}
	
	private void setCurrentWorld(String name) throws IOException {
		if(name == null) {
			if(storageManager == null)
				return;
			saveQueue.clear();
			loadQueue.clear();
			mergeQueue.clear();
			drainWorkQueues();
			while(NodeImageData.allNIDs.size() > 0)
				NodeImageData.allNIDs.get(0).destroy();
			storageManager.close();
			drainWorkQueues(); // storageManager.close() queues a task on the I/O thread
			storageManager = null;
		} else {
			if(storageManager != null)
				throw new RuntimeException("A world is already open.");
			
			final File mcDir = (File)FMLInjectionData.data()[6];
			storageManager = new StorageManager(mcDir, name);
		}
	}

	@SubscribeEvent
	public void onTick(TickEvent.ClientTickEvent evt) throws Exception {
		
		if(mainThread == null)
			mainThread = Thread.currentThread();
		
		if(multiSampleLevel == Integer.MIN_VALUE) {
			int maxTextureSize = glGetInteger(GL_MAX_TEXTURE_SIZE);
			multiSampleLevel = MAX_MULTISAMPLE_LEVEL;
			while((IMAGE_DIMENSION << multiSampleLevel) * InfiViewCapturer.IMAGES_IN_ARRAY_BY_DIMENSION > maxTextureSize) {
				multiSampleLevel--;
				// TODO: If multiSampleLevel < 0, then use smaller images, or disable mod. Or split arrays up into multiple smaller textures. Either way, don't just crash.
				if(multiSampleLevel < 0)
					throw new RuntimeException("Your graphics card's maximum texture size ("+maxTextureSize+") is not sufficient for InfiView. At least "+(IMAGE_DIMENSION * InfiViewCapturer.IMAGES_IN_ARRAY_BY_DIMENSION)+" is required");
			}
			
			if(multiSampleLevel < MAX_MULTISAMPLE_LEVEL)
				LOGGER.log(Level.WARN, "Multi-sample level set to "+multiSampleLevel+" when default is "+MAX_MULTISAMPLE_LEVEL+" due to texture size restrictions. Maximum texture dimension is "+maxTextureSize+" pixels.");
			else
				LOGGER.log(Level.INFO, "Multi-sample level set to "+multiSampleLevel+".");
			
			GLDebugOutputLogger.init();
		}
		
		if(storageManager == null)
			return;
		
		NodeImageData.expiryTickCounter++;
		NodeImageData.checkNIDExpiry();
		
		while(true) {
			Runnable cb = mainThreadCallbackQueue.poll();
			if(cb == null)
				break;
			cb.run();
		}
		
		// Order of these matters very slightly; if a load operation moves from IO to CPU state
		// in between these two reads, it'll be double-counted rather than not counted.
		int load_IO = ResourceAllocator.loadOperationsAwaitingIO.get();
		int load_CPU = ResourceAllocator.loadOperationsAwaitingCPU.get();
		int maxLoadsToStart = Math.min(loadQueue.size(), Math.min(30 - load_IO - load_CPU, 20 - load_IO));
		for(int k = 0; k < maxLoadsToStart; k++)
			startALoad();
		
		int save_download = ResourceAllocator.saveOperationsAwaitingDownload.get();
		int save_CPU = ResourceAllocator.saveOperationsAwaitingCPU.get();
		int save_IO = ResourceAllocator.saveOperationsAwaitingIO.get();
		int maxSavesToStart = Math.min(saveQueue.size(), Math.min(4 - save_IO - save_CPU - save_download,
			Math.min(3 - save_CPU - save_download, 2 - save_download)));
		for(int k = 0; k < maxSavesToStart; k++)
			startASave();
		
		int merge_all = ResourceAllocator.mergeOperationsRunning.get();
		int maxMergesToStart = Math.min(mergeQueue.size(), 3 - merge_all);
		for(int k = 0; k < maxMergesToStart; k++)
			startAMerge();
		
		textureDownloadQueue.tick(false);
	}
	
	static List<PendingLoadOperation> loadQueue = new ArrayList<>();
	private static void startALoad() {
		if(loadQueue.isEmpty())
			return;
		PendingLoadOperation op = loadQueue.remove(loadQueue.size() - 1);
		op.start();
	}
	
	static LinkedHashSet<WorldRenderer> saveQueue = new LinkedHashSet<>();
	
	private static void startASave() {
		if(saveQueue.isEmpty())
			return;
		
		WorldRenderer r = saveQueue.iterator().next();
		saveQueue.remove(r);
		
		// TODO: ensure renderer hasn't been destroyed
		if(!isEmptyRenderer(r)) {
			CachedQuadTreeNode node = storageManager.getNode(r.posX>>4, r.posZ>>4);
			ChunkCoordinates coords = new ChunkCoordinates(r.posX>>4, r.posY>>4, r.posZ>>4);
			PendingSaveOperation op = new PendingSaveOperation(node, coords, r);
			op.start();
		}
	}
	
	// TODO: use something fairer. Maybe: keep a counter, get the counter'th item and replace it with the last, increment counter, and wrap the counter when it gets to the end.
	public static List<PendingMergeOperation> mergeQueue = new ArrayList<>(); 
	private static void startAMerge() {
		if(mergeQueue.isEmpty())
			return;
		PendingMergeOperation op = mergeQueue.remove(mergeQueue.size() - 1);
		op.start();
	}

	private static boolean isEmptyRenderer(WorldRenderer r) {
		for(boolean b : r.skipRenderPass)
			if(!b)
				return false;
		return true;
	}
	
	@SubscribeEvent(priority=EventPriority.LOWEST)
	public void onRenderPost(RenderWorldEvent.Post evt) {
		if(!ENABLE_SAVING)
			return;
		
		if(isEmptyRenderer(evt.renderer)) {
			CachedQuadTreeNode node = storageManager.getNode(evt.renderer.posX>>4, evt.renderer.posZ>>4);
		    node.deleteImageAndMarkParentsForUpdate(evt.renderer.posY>>4);
		} else {
			saveQueue.add(evt.renderer);
		}
	}

	private static int[][] yChunkOrder = new int[16][16];
	static {
		for(int playerY = 0; playerY < 16; playerY++) {
			int pos = 0;
			for(int y = 0; y < playerY; y++)
				yChunkOrder[playerY][pos++] = y;
			for(int y = 15; y >= playerY; y--)
				yChunkOrder[playerY][pos++] = y;
			assert pos == 16;
		}
	}
	
	private void renderDistantChunks(CachedQuadTreeNode node, int x, int z, int scale, double camx, double camy, double camz, int targetNodeScale) throws ReflectiveOperationException {
		double chunkCentreX = (16.0*x + 8.0*(1<<scale));
		double chunkCentreZ = (16.0*z + 8.0*(1<<scale));
		
		int numYChunks = Math.max(1, 16 >> scale);
		
		if(node.loadedImageData == null)
			node.loadedImageData = new NodeImageData(node);
		NodeImageData nodeImageData = node.loadedImageData;
		
		nodeImageData.lastAccessTime = NodeImageData.expiryTickCounter;
		
		node.queueMergeIfNeeded();
		
		short yChunkMask;
		if(scale == 0) {
			yChunkMask = 0;
			for(int y = 0; y < 16; y++) {
				if(!isRealRenderingAvailable(x, y, z)) {
					yChunkMask |= 1 << y;
				}
			}
			if(yChunkMask == 0)
				return; // Nothing to render for this chunk!
		} else {
			yChunkMask = (short)((1 << numYChunks) - 1);
		}
		
		int playerY = (int)camy * numYChunks / 256;
		for(int y = 0; y < numYChunks; y++)	{
			if((yChunkMask & (1 << y)) == 0)
				continue;
			
			float dx = (float)(chunkCentreX-camx);
			float dy = (float)((y+0.5f)*256/numYChunks-camy);
			float dz = (float)(chunkCentreZ-camz);
			
			int direction = InfiViewDirections.getClosestDirection(dx, dy, dz);
			
			nodeImageData.checkMinichunkDirection(y, direction);
		}
		
		yChunkMask &= nodeImageData.getImagePresentMask();
		if(yChunkMask == 0)
			return;
		
		glBindTexture(GL_TEXTURE_2D, nodeImageData.getGLTexture());
		glBegin(GL_QUADS);
		
		// TODO: make this a user-toggleable option for debugging purposes
		if(false)
		switch(scale) {
		case 0: glColor3f(1, 0.5f, 0.5f); break;
		case 1: glColor3f(0.5f, 1, 0.5f); break;
		case 2: glColor3f(0.5f, 0.5f, 1); break;
		case 3: glColor3f(1, 1, 0); break;
		case 4: glColor3f(1, 0, 1); break;
		case 5: glColor3f(0, 1, 1); break;
		default: glColor3f(1, 1, 1); break;
		}
		
		Vector3f centre = new Vector3f();
		Vector3f udir = new Vector3f();
		Vector3f vdir = new Vector3f();
		
		for(int y : yChunkOrder[playerY]) {
			if((yChunkMask & (1 << y)) == 0)
				continue;
			
			int direction = nodeImageData.getCurrentDirection(y);
			if(direction == -1)
				continue;
			
			// TODO remove commented code
			//int chunkYDist = Math.abs(ve.chunkCoordY - y);
			
			float v1 = y/(float)numYChunks;
			float v2 = (y+1)/(float)numYChunks;
			
			centre.set((float)(chunkCentreX-camx), (float)((y+0.5f)*256/numYChunks-camy), (float)(chunkCentreZ-camz));
			
			InfiViewDirections.getDisplayDir(direction, udir, vdir);
			
			float imageScale = InfiViewDirections.getImageDimensionIngame(direction) * (1 << scale);
			udir.scale(imageScale);
			vdir.scale(imageScale);
			
			glTexCoord2f(0, v2); glVertex3f(centre.x - udir.x + vdir.x, centre.y - udir.y + vdir.y, centre.z - udir.z + vdir.z);
			glTexCoord2f(0, v1); glVertex3f(centre.x - udir.x - vdir.x, centre.y - udir.y - vdir.y, centre.z - udir.z - vdir.z);
			glTexCoord2f(1, v1); glVertex3f(centre.x + udir.x - vdir.x, centre.y + udir.y - vdir.y, centre.z + udir.z - vdir.z);
			glTexCoord2f(1, v2); glVertex3f(centre.x + udir.x + vdir.x, centre.y + udir.y + vdir.y, centre.z + udir.z + vdir.z);
		}
		glEnd();
	}
	
	private static Field field_RenderGlobal_worldRenderers;
	private static Field field_WorldRenderer_isInitialized;
	static {
		try {
			boolean sawOne = false;
			for(Field f : RenderGlobal.class.getDeclaredFields()) {
				// Second field of type WorldRenderer[]
				if(f.getType() == WorldRenderer[].class) {
					if(!sawOne)
						sawOne = true;
					else {
						f.setAccessible(true);
						field_RenderGlobal_worldRenderers = f;
						break;
					}
				}
			}
			if(field_RenderGlobal_worldRenderers == null)
				throw new RuntimeException("Couldn't find worldRenderers field in RenderGlobal");
			
			for(Field f : WorldRenderer.class.getDeclaredFields()) {
				// First private boolean field
				if(f.getType() == Boolean.TYPE && Modifier.isPrivate(f.getModifiers())) {
					f.setAccessible(true);
					field_WorldRenderer_isInitialized = f;
					break;
				}
			}
			if(field_WorldRenderer_isInitialized == null)
				throw new RuntimeException("Couldn't find isInitialized field in WorldRenderer");
		} catch(Exception e) {
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Returns true if the specified size-1 minichunk is being rendered through
	 * Minecraft's normal mechanism. 
	 */
	private boolean isRealRenderingAvailable(int x, int y, int z) throws ReflectiveOperationException {
		RenderGlobal rg = Minecraft.getMinecraft().renderGlobal;
		
		int diameter = Minecraft.getMinecraft().gameSettings.renderDistanceChunks * 2 + 1;		
		// These fields in RenderGlobal are private, but their values are easily calculated.
		int renderChunksWide = diameter;
		int renderChunksTall = 16;
		int renderChunksDeep = diameter;
		
		int xmod = x % renderChunksWide;
		int ymod = y % renderChunksTall;
		int zmod = z % renderChunksDeep;
		if(xmod < 0) xmod += renderChunksWide;
		if(ymod < 0) ymod += renderChunksTall;
		if(zmod < 0) zmod += renderChunksDeep;
		
		WorldRenderer[] renderers = (WorldRenderer[])field_RenderGlobal_worldRenderers.get(rg);
		
		WorldRenderer r = renderers[(zmod * renderChunksTall + ymod) * renderChunksWide + xmod];
		
		return r.posX == x*16 && r.posY == y*16 && r.posZ == z*16 && ((Boolean)field_WorldRenderer_isInitialized.get(r)).booleanValue();
	}

	@SubscribeEvent
	public void renderDistantChunks(RenderWorldLastEvent evt) {
		Entity ve = Minecraft.getMinecraft().renderViewEntity;
		if(ve == null)
			return;
		
		if(storageManager == null)
			return;
		
		final double camx = ve.lastTickPosX + (ve.posX - ve.lastTickPosX) * evt.partialTicks;
		final double camy = ve.lastTickPosY + (ve.posY - ve.lastTickPosY) * evt.partialTicks;
		final double camz = ve.lastTickPosZ + (ve.posZ - ve.lastTickPosZ) * evt.partialTicks;
		
		// To make alpha blending work, we use the quadtree structure to render back-to-front.
		// This means we don't actually need the depth buffer.
		// We still need to depth-test against real terrain though, so we leave GL_DEPTH_TEST enabled.
		glDepthMask(false);
		
		// Drawing with premultiplied alpha
		glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
		
		glEnable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		
		glColor4f(1, 1, 1, 1);
		
		glDisable(GL_CULL_FACE);
		
		// TODO: use a VBO
		
		final int camChunkX = (int)Math.floor(camx / 16);
		final int camChunkZ = (int)Math.floor(camz / 16);
		
		//glDisable(GL_CULL_FACE);
		//glPolygonMode(GL_BACK, GL_LINE);
		//glDisable(GL_TEXTURE_2D);
		
		// We render back-to-front to allow alpha transparency to work.
		// TODO: render Y-levels back-to-front as well. (Starting from the top and bottom, converging on the camera position), in case the player is underground.
		storageManager.visitQuadTreeBackToFront(camChunkX, camChunkZ, new QuadTreeVisitor() {
			@Override
			public QuadTreeVisitResult visit(int x, int z, int scale, CachedQuadTreeNode node) {
				// TODO: make sure this works correctly
				int chunkXDist = InfiViewMathUtils.distanceBetweenLineSegments(x, x+(1<<scale), camChunkX, camChunkX+1);
				int chunkZDist = InfiViewMathUtils.distanceBetweenLineSegments(z, z+(1<<scale), camChunkZ, camChunkZ+1);
				int horizDistanceFromPlayer = Math.max(chunkXDist, chunkZDist);
				
				int targetNodeScale = Math.max(0, (32 - Integer.numberOfLeadingZeros(horizDistanceFromPlayer))/4);
				
				if(scale > targetNodeScale)
					return QuadTreeVisitResult.VISIT_CHILDREN;
				
				try {
					renderDistantChunks(node, x, z, scale, camx, camy, camz, targetNodeScale);
				} catch (ReflectiveOperationException e) {
					e.printStackTrace();
				}
				
				return QuadTreeVisitResult.SKIP_CHILDREN;
			}
		});
		
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glDepthMask(true);
		glEnable(GL_CULL_FACE);
	}

	public static void executeOnMainThread(Runnable task) {
		if(Thread.currentThread() == mainThread)
			task.run();
		else
			mainThreadCallbackQueue.add(task);
	}
}
