package mods.immibis.infiview;

import static org.lwjgl.opengl.GL11.*;
import static mods.immibis.infiview.InfiViewMod.*;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import mods.immibis.infiview.storage.CachedQuadTreeNode;

import org.lwjgl.opengl.GL12;

/**
 * Holds all the in-memory image data for one chunk.
 * This is a single texture, containing a vertical column of 16 images - one for each minichunk.
 * As the player's direction to the chunk changes, we asynchronously load different images from disk
 * into the that texture.
 * 
 * All of this class's methods are only called on the render thread, except where otherwise specified.
 */
public final class NodeImageData {
	// Holds the currently loaded direction index for each minichunk, or -1 if none is loaded.
	private final byte[] directions;
	
	private short loadingMask = 0; // Bit is set for each minichunk which is being loaded.
	private short imagePresentMask = 0; // Bit is set if the corresponding minichunk has an image (its image is not filled with garbage) 
	private int glTexture = 0;
	private final CachedQuadTreeNode node;
	private boolean destroyed;
	
	public static int expiryTickCounter;
	
	public int lastAccessTime; // Value of expiryTickCounter when this node was last used.
	
	public static List<NodeImageData> allNIDs = new ArrayList<>();
	private int myIndexInAllNIDs;
	
	public NodeImageData(CachedQuadTreeNode node) {
		this.node = node;
		
		int numMinichunks = Math.max(1, 16 / node.getSize());
		
		assert (numMinichunks & (numMinichunks - 1)) == 0 : "numMinichunks must be a power of two";
		
		directions = new byte[numMinichunks];
		Arrays.fill(directions, (byte)-1);
		
		myIndexInAllNIDs = allNIDs.size();
		allNIDs.add(this);
	}
	
	public void destroy() {
		assert !destroyed;
		if(glTexture != 0)
			glDeleteTextures(glTexture);
		glTexture = -1;
		assert node.loadedImageData == this;
		node.loadedImageData = null;
		destroyed = true;
		
		assert allNIDs.get(myIndexInAllNIDs) == this;
		
		// Swap with last element, then remove last element.
		int lastIndex = allNIDs.size() - 1;
		NodeImageData lastNID = allNIDs.remove(lastIndex);
		if(myIndexInAllNIDs != lastIndex) {
			allNIDs.set(myIndexInAllNIDs, lastNID);
			lastNID.myIndexInAllNIDs = myIndexInAllNIDs;
		} else {
			assert lastNID == this;
		}
	}

	public int getGLTexture() {
		assert !destroyed;
		if(glTexture == 0) {
			glTexture = glGenTextures();
			glBindTexture(GL_TEXTURE_2D, glTexture);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_BASE_LEVEL, 0);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_MAX_LEVEL, 0);
			glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, IMAGE_DIMENSION, directions.length*IMAGE_DIMENSION, 0, GL_RGBA, GL_UNSIGNED_BYTE, (ByteBuffer)null);
		}
		return glTexture;
	}

	/**
	 * Checks that the given direction's image is loaded for the given minichunk.
	 * If not, queues an asynchronous request to load it, unless one is already queued.
	 * This may make OpenGL calls directly, so is not safe between glBegin and glEnd.
	 * (TODO: it no longer makes OpenGL calls directly, so remove this warning?)
	 */
	public void checkMinichunkDirection(final int y, final int direction) {
		assert !destroyed;
		byte oldDir = directions[y];
		if(oldDir == direction)
			return;
		
		if((loadingMask & (1 << y)) != 0)
			return; // already loading
		loadingMask |= (1 << y);
		
		InfiViewMod.loadQueue.add(new PendingLoadOperation(node, y, direction));
	}

	void doneLoadingImage(int y, int direction, boolean wasPresent) {
		if(destroyed)
			return;
		
		assert (loadingMask & (1 << y)) != 0 : "Never started loading this image!";
		
		loadingMask &= ~(1 << y);
		directions[y] = (byte)direction;
		
		if(wasPresent)
			imagePresentMask |= 1 << y;
		else
			imagePresentMask &= ~(1 << y);
	}

	public int getCurrentDirection(int y) {
		assert !destroyed;
		return directions[y];
	}

	public short getImagePresentMask() {
		assert !destroyed;
		return imagePresentMask;
	}

	public void markMinichunkForReload(int y) {
		assert !destroyed;
		directions[y] = -1;
		imagePresentMask &= ~(1 << y);
	}

	private static int nidExpiryScanPos = 0;
	public static void checkNIDExpiry() {
		if(allNIDs.size() == 0)
			return;
		nidExpiryScanPos = (nidExpiryScanPos + 1) % allNIDs.size();
		NodeImageData d = allNIDs.get(nidExpiryScanPos);
		if(d.lastAccessTime < expiryTickCounter - 400) // 400 ticks = 20 seconds
			d.destroy();
	}
}
