package mods.immibis.infiview;

import static mods.immibis.infiview.InfiViewMod.*;
import static org.lwjgl.opengl.GL11.*;

import java.nio.ByteBuffer;

import org.apache.logging.log4j.Level;

import mods.immibis.infiview.storage.CachedQuadTreeNode;
import mods.immibis.infiview.storage.ImageDataFile2;
import mods.immibis.infiview.storage.ImageJPEGCompressor;

public class PendingLoadOperation {
	
	private final CachedQuadTreeNode node;
	private final int y; 
	private final int direction;
	private final NodeImageData nodeImageData;
	
	public PendingLoadOperation(CachedQuadTreeNode node, int y, int direction) {
		this.node = node;
		this.y = y;
		this.direction = direction;
		this.nodeImageData = node.loadedImageData;
	}
	
	private int imageSeriesRefPoint;
	private long imageSeriesID;
	private ByteBuffer imageSeriesBuffer;
	private ImageDataFile2 dataFile;
	
	public void start() {
		if(nodeImageData != node.loadedImageData)
			return;
		
		try {
			imageSeriesRefPoint = node.getImageSeriesRefPoint(y);
			imageSeriesID = InfiViewMod.storageManager.quadTreeFile.getImageSeries(imageSeriesRefPoint);
			if(imageSeriesID < 0) {
				InfiViewMod.LOGGER.log(Level.TRACE, "Clearing minichunk {}, node {}", y, node.getNodeID());
				nodeImageData.doneLoadingImage(y, direction, false);
				return;
			}
		} catch(Throwable t) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Failed to load minichunk {}, node {}", y, node.getNodeID());
			nodeImageData.doneLoadingImage(y, direction, false);
			return;
		}
		
		imageSeriesBuffer = InfiViewMod.renderchunkImageBuffers.getBuffer();
		InfiViewMod.LOGGER.log(Level.TRACE, "Reading image series {}, with buffer {}", imageSeriesID, System.identityHashCode(imageSeriesBuffer));
		
		dataFile = InfiViewMod.storageManager.imageDataFile2;
		
		ResourceAllocator.loadOperationsAwaitingIO.incrementAndGet();
		
		InfiViewMod.ioThread.enqueue(this);
	}
	
	public void doIOLoadNow() {
		try {
			final int compressedDataSize = dataFile.getBlockSize(imageSeriesID);
			imageSeriesBuffer.position(0).limit(compressedDataSize);
			dataFile.read(imageSeriesBuffer, dataFile.getFilePosition(imageSeriesID));
			
			imageSeriesBuffer.position(0).limit(compressedDataSize);
			ResourceAllocator.loadOperationsAwaitingCPU.incrementAndGet();
			InfiViewMod.cpuThread.enqueue(this);
		} catch(Throwable e) {
			InfiViewMod.renderchunkImageBuffers.returnBuffer(imageSeriesBuffer);
			InfiViewMod.LOGGER.log(Level.ERROR, "Error loading an image", e);
		} finally {
			ResourceAllocator.loadOperationsAwaitingIO.decrementAndGet();
		}
	}
	
	public void doDecompressNow() {
		try {
			ImageJPEGCompressor.decompressImage(imageSeriesBuffer);
		} catch(Exception e) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Error decompressing an image", e);
			InfiViewMod.mainThreadCallbackQueue.add(new Runnable() {
				@Override
				public void run() {
					// No-op if nodeImageData has been destroyed
					nodeImageData.doneLoadingImage(y, direction, false);
				}
			});
			return;
		} finally {
			ResourceAllocator.loadOperationsAwaitingCPU.decrementAndGet();
		}
		
		InfiViewMod.mainThreadCallbackQueue.add(new Runnable() {
			@Override
			public void run() {
				doUploadNow();
			}
		});
	}
	
	void doUploadNow() {
		if(nodeImageData == node.loadedImageData) {
			InfiViewMod.LOGGER.log(Level.TRACE, "Uploading an image for minichunk {}, node {}", y, node.getNodeID());
			imageSeriesBuffer.limit((direction+1)*IMAGE_BYTES).position(direction*IMAGE_BYTES);
			glBindTexture(GL_TEXTURE_2D, node.loadedImageData.getGLTexture());
			glTexSubImage2D(GL_TEXTURE_2D, 0, 0, y*IMAGE_DIMENSION, IMAGE_DIMENSION, IMAGE_DIMENSION, GL_RGBA, GL_UNSIGNED_BYTE, imageSeriesBuffer);
			node.loadedImageData.doneLoadingImage(y, direction, true);
		}
		InfiViewMod.renderchunkImageBuffers.returnBuffer(imageSeriesBuffer);
	}
}
