package mods.immibis.infiview;

import static mods.immibis.infiview.InfiViewMod.*;
import static org.lwjgl.opengl.ARBFramebufferObject.*;
import static org.lwjgl.opengl.ARBPixelBufferObject.*;
import static org.lwjgl.opengl.GL11.*;
import static org.lwjgl.opengl.GL15.*;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;

import org.apache.logging.log4j.Level;
import org.lwjgl.opengl.Display;
import org.lwjgl.opengl.GL11;
import org.lwjgl.opengl.GL12;
import org.lwjgl.opengl.GL14;
import org.lwjgl.util.vector.Vector3f;

import mods.immibis.infiview.capture.CaptureUtils;
import mods.immibis.infiview.capture.InfiViewCapturer;
import mods.immibis.infiview.storage.CachedQuadTreeNode;
import mods.immibis.infiview.storage.ImageDataFile2;
import mods.immibis.infiview.storage.ImageJPEGCompressor;

/**
 * For a merge operation, we allocate 4 or 8 image series buffers.
 * We then loop over all the input image series, loading them and decompressing them.
 * We then render, download, compress and save.
 * (Then we mark the node as no longer needing a merge).
 * 
 * TODO: If the node's children get updated while the merge is running, the last step will erroneously
 * mark the node as not needing a merge even though it needs to be merged again.
 */
public class PendingMergeOperation {

	private final CachedQuadTreeNode node;
	private final int y;
	private final int numInputs;
	private long[] inputBlockNums;
	private ByteBuffer[] inputBuffers;
	private int pbo;
	private int outputTex;
	private boolean isSaving;
	private ByteBuffer outputImageSeriesBuffer;
	private byte[] outputCompressedImage;
	
	
	public PendingMergeOperation(CachedQuadTreeNode node, int y) {
		this.node = node;
		this.y = y;
		this.numInputs = node.getSize() <= InfiViewMod.NUM_RENDERCHUNKS_PER_CHUNK ? 8 : 4;
		this.inputBlockNums = new long[numInputs];
		this.inputBuffers = new ByteBuffer[numInputs];
	}
	
	
	public void start() {
		ResourceAllocator.mergeOperationsRunning.incrementAndGet();

		for(int k = 0; k < numInputs; k++) {
			int childNumber = k&3;
			int childY = y*2 + ((k & 4) != 0 ? 1 : 0);
			CachedQuadTreeNode childNode = node.getChild(childNumber);
			if(childNode == null)
				inputBlockNums[k] = -1;
			else
				inputBlockNums[k] = InfiViewMod.storageManager.quadTreeFile.getImageSeries(childNode.getImageSeriesRefPoint(childY));
		}
		
		InfiViewMod.ioThread.enqueue(this);
	}
	
	public void doIOOperationNow() {
		if(!isSaving) {
			
			ImageDataFile2 file = InfiViewMod.storageManager.imageDataFile2;
			
			for(int k = 0; k < numInputs; k++) {
				long blockNum = inputBlockNums[k];
				if(blockNum == -1) {
					inputBuffers[k] = null;
					continue;
				}
				int size = file.getBlockSize(blockNum);
				if(size > RENDERCHUNK_IMAGE_BYTES || size < 0) {
					inputBuffers[k] = null;
					continue;
				}
				inputBuffers[k] = InfiViewMod.renderchunkImageBuffers.getBuffer();
				inputBuffers[k].position(0).limit(size);
				try {
					file.read(inputBuffers[k], file.getFilePosition(blockNum));
					inputBuffers[k].position(0).limit(size);
				} catch (IOException e) {
					InfiViewMod.LOGGER.log(Level.ERROR, "Failed to read image in block "+blockNum+" for merge", e);
					InfiViewMod.renderchunkImageBuffers.returnBuffer(inputBuffers[k]);
					inputBuffers[k] = null;
				}
			}
			
			InfiViewMod.cpuThread.enqueue(this);
			
		} else { // if(isSaving)
			
			node.writeCompressedImageSeriesNow(y, outputCompressedImage);
			InfiViewMod.LOGGER.log(Level.TRACE, "Wrote an image series from a merge.");
			ResourceAllocator.mergeOperationsRunning.decrementAndGet();
			
			InfiViewMod.executeOnMainThread(new Runnable() {
				@Override
				public void run() {
					if(node.loadedImageData != null)
						node.loadedImageData.markMinichunkForReload(y);
					node.onDoneMerging(y);
				}
			});
		}
	}
	
	public void doCPUOperationNow() {
		if(!isSaving) {
			for(int k = 0; k < numInputs; k++) {
				try {
					if(inputBuffers[k] != null)
						ImageJPEGCompressor.decompressImage(inputBuffers[k]);
				} catch(Exception e) {
					InfiViewMod.LOGGER.log(Level.ERROR, "Failed to decompress an image for merge", e);
					InfiViewMod.renderchunkImageBuffers.returnBuffer(inputBuffers[k]);
					inputBuffers[k] = null;
				}
			}
			
			InfiViewMod.mainThreadCallbackQueue.add(new Runnable() {
				@Override
				public void run() {
					doRenderOperationNow();
				}
			});
			
		} else { // if(isSaving)
			
			outputCompressedImage = ImageJPEGCompressor.compressImage(outputImageSeriesBuffer, InfiViewMod.IMAGE_DIMENSION, InfiViewMod.IMAGE_DIMENSION*InfiViewDirections.NUM_DIRECTIONS);
			InfiViewMod.ioThread.enqueue(this);
		}
	}
	
	public void doRenderOperationNow() {
		outputTex = GL11.glGenTextures();
		combineImages(inputBuffers, outputTex);
		
		for(ByteBuffer b : inputBuffers)
			if(b != null)
				InfiViewMod.renderchunkImageBuffers.returnBuffer(b);
		Arrays.fill(inputBuffers, null);
			
		InfiViewMod.textureDownloadQueue.enqueue(this);
	}
	
	public void doDownloadNow() {
		pbo = InfiViewMod.arrayImagePBOs.getBuffer();
		glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, pbo);
		glBufferData(GL_PIXEL_PACK_BUFFER_ARB, (long)(InfiViewCapturer.ARRAY_TEXTURE_DIMENSION*InfiViewCapturer.ARRAY_TEXTURE_DIMENSION*4), GL_STREAM_READ);
		// TODO: use when supported instead of glBufferData.
		//ARBBufferStorage.glBufferStorage(GL_PIXEL_PACK_BUFFER_ARB, (long)dataSize, GL30.GL_MAP_READ_BIT | ARBBufferStorage.GL_CLIENT_STORAGE_BIT);
		glBindTexture(GL_TEXTURE_2D, outputTex);
		
		glGetTexImage(GL_TEXTURE_2D, multiSampleLevel, GL_RGBA, GL_UNSIGNED_BYTE, 0L);
		
		glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, 0);
	}
	
	private static ByteBuffer mappedBuffer = null;
	public void onDownloadCompleted() {
		glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, pbo);
		mappedBuffer = glMapBuffer(GL_PIXEL_PACK_BUFFER_ARB, GL_READ_ONLY, mappedBuffer);
		
		// First, we reorder the data, from a NxN array of textures, into an (N*N)x1 array. (Actually a Mx1 array where M <= N*N)
		// imageSeriesBuffer is the buffer we store the reordered data into.
		
		outputImageSeriesBuffer = InfiViewMod.renderchunkImageBuffers.getBuffer();
		CaptureUtils.reorderImageSeries(mappedBuffer, outputImageSeriesBuffer);
		glUnmapBuffer(GL_PIXEL_PACK_BUFFER_ARB);
		glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, 0);
		glDeleteTextures(outputTex);
		InfiViewMod.arrayImagePBOs.returnBuffer(pbo);
		
		isSaving = true;
		InfiViewMod.cpuThread.enqueue(this);
	}
	


	private static int lastTextureSize = -1;
	private static int framebuffer = -1;
	private static int depthRenderbuffer = -1;
	
	private static void init(int textureSize) {
		if(framebuffer == -1) {
			framebuffer = glGenFramebuffers();
			depthRenderbuffer = glGenRenderbuffers();
		}
		
		if(lastTextureSize != textureSize) {
	    	glBindRenderbuffer(GL_RENDERBUFFER, depthRenderbuffer);
	    	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, textureSize, textureSize);
	    	
	    	glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
	    	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, depthRenderbuffer);
	    	
	    	lastTextureSize = textureSize;
	    } else {
	    	glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
	    }
    	
	}
	
	/**
	 * Combines all the given images into one.
	 * This is sensitive to the order of images, which should be
	 * -X-Z, +X-Z, -X+Z, +X+Z. If 8 images are given, then bottom layer first (4 images), followed by top layer's 4 images.
	 * All the passed buffers will be freed.
	 */
	private static void combineImages(ByteBuffer[] imageSeriesBuffers, int outputTex) {
		assert Thread.currentThread() == InfiViewMod.mainThread;
		
		// We create an output texture to hold an image series.
		// We generate a texture to hold 8 (or 4) images, then for each direction
		// we upload the images for that direction, render them, and download the result.
		
		final int ssaaLevel = InfiViewMod.multiSampleLevel;
		assert ssaaLevel >= 0;
		
		int renderImageSize = IMAGE_DIMENSION << ssaaLevel;
		
		int outputTextureSize = InfiViewCapturer.IMAGES_IN_ARRAY_BY_DIMENSION*renderImageSize;
		init(outputTextureSize);
		
		// TODO change to false, remove. (For debugging only)
     	final boolean RENDER_TO_WINDOW_INSTEAD = false;
     	final boolean RENDER_TO_WINDOW_AS_WELL = false;
		
		if(true) {
			// TODO: reuse these textures?
			int inputTex = glGenTextures();
		
			glBindTexture(GL_TEXTURE_2D, inputTex);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_BASE_LEVEL, 0);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_MAX_LEVEL, 0);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
			glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, InfiViewMod.IMAGE_DIMENSION, InfiViewMod.IMAGE_DIMENSION*imageSeriesBuffers.length, 0, GL_RGBA, GL_UNSIGNED_BYTE, (ByteBuffer)null);
			
			glBindTexture(GL_TEXTURE_2D, outputTex);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_BASE_LEVEL, 0);
			glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_MAX_LEVEL, ssaaLevel);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
			glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, outputTextureSize, outputTextureSize, 0, GL_RGBA, GL_UNSIGNED_BYTE, (ByteBuffer)null);
			
			if(RENDER_TO_WINDOW_INSTEAD) {
				glBindFramebuffer(GL_FRAMEBUFFER, 0);
			} else {
				glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
				glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, outputTex, 0);
			}
			
			glClearColor(0, 0, 0, 0);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			glColor3f(1, 1, 1);
			
			// GL_COLOR_BUFFER_BIT for blend function
			glPushAttrib(GL_VIEWPORT_BIT | GL_COLOR_BUFFER_BIT);
			
	        glBindTexture(GL_TEXTURE_2D, inputTex);
	        glEnable(GL_TEXTURE_2D);
	        glEnable(GL_BLEND);
	        
	        // Input and output are both premultiplied.
	        // TODO review this
	        GL14.glBlendFuncSeparate(GL_ONE, GL_ONE_MINUS_SRC_ALPHA, GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
	        
	        for(int direction = 0; direction < InfiViewDirections.NUM_DIRECTIONS; direction++) {
				for(int image = 0; image < imageSeriesBuffers.length; image++) {
					if(imageSeriesBuffers[image] == null)
						continue;
					
					ByteBuffer isb = imageSeriesBuffers[image];
					isb.limit((direction+1)*InfiViewMod.IMAGE_BYTES).position(direction*InfiViewMod.IMAGE_BYTES);
					glTexSubImage2D(GL_TEXTURE_2D, 0, 0, image*IMAGE_DIMENSION, IMAGE_DIMENSION, IMAGE_DIMENSION, GL_RGBA, GL_UNSIGNED_BYTE, isb);
				}
				
				
				glMatrixMode(GL_PROJECTION);
				glLoadIdentity();
				double orthoSize = InfiViewDirections.getImageDimensionIngame(direction) * 2;
				glOrtho(-orthoSize, orthoSize, -orthoSize, orthoSize, -100000, 100000);
				glMatrixMode(GL_MODELVIEW);
				glLoadIdentity();
				
				int imx = direction % InfiViewCapturer.IMAGES_IN_ARRAY_BY_DIMENSION;
				int imy = direction / InfiViewCapturer.IMAGES_IN_ARRAY_BY_DIMENSION;
				glViewport(renderImageSize*imx, renderImageSize*imy, renderImageSize, renderImageSize);
				InfiViewDirections.setGLViewAngleByDirection(direction);
				
				
				glBegin(GL_QUADS);
				
				Vector3f centre = new Vector3f();
				Vector3f udir = new Vector3f();
				Vector3f vdir = new Vector3f();
				
				// TODO: render back-to-front!!!
				
				for(int image = 0; image < imageSeriesBuffers.length; image++) {
					if(imageSeriesBuffers[image] == null)
						continue;
					
					float v1 = image/(float)imageSeriesBuffers.length;
					float v2 = (image+1)/(float)imageSeriesBuffers.length;
					
					centre.set((image & 1) == 0 ? -8 : 8,
						imageSeriesBuffers.length == 4 ? 0 : (image & 4) == 0 ? -8 : 8,
						(image & 2) == 0 ? -8 : 8);
					
					InfiViewDirections.getDisplayDir(direction, udir, vdir);
					
					float scale = InfiViewDirections.getImageDimensionIngame(direction);
					udir.scale(scale);
					vdir.scale(scale);
					
					glTexCoord2f(0, v2); glVertex3f(centre.x - udir.x + vdir.x, centre.y - udir.y + vdir.y, centre.z - udir.z + vdir.z);
					glTexCoord2f(0, v1); glVertex3f(centre.x - udir.x - vdir.x, centre.y - udir.y - vdir.y, centre.z - udir.z - vdir.z);
					glTexCoord2f(1, v1); glVertex3f(centre.x + udir.x - vdir.x, centre.y + udir.y - vdir.y, centre.z + udir.z - vdir.z);
					glTexCoord2f(1, v2); glVertex3f(centre.x + udir.x + vdir.x, centre.y + udir.y + vdir.y, centre.z + udir.z + vdir.z);
				}
				glEnd();
			}
	        
	        glPopAttrib();
			
			glBindFramebuffer(GL_FRAMEBUFFER, 0);
			
	        if(RENDER_TO_WINDOW_INSTEAD) {
				try {
					Display.update();
					Thread.sleep(200);
				} catch(Exception e) {
					e.printStackTrace();
				}
			}
			
			glBindTexture(GL_TEXTURE_2D, outputTex);
			glEnable(GL_TEXTURE_2D); // ATI driver bug; glGenerateMipmaps requires GL_TEXTURE_2D enabled
			glGenerateMipmap(GL_TEXTURE_2D);
			
			if(RENDER_TO_WINDOW_AS_WELL) {
				try {
					glClearColor(1,1,0.5f,1);
					glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
					glMatrixMode(GL_PROJECTION);
					glLoadIdentity();
					//glOrtho(-1, 1, -1, 1, -1, 1);
					glMatrixMode(GL_MODELVIEW);
					glLoadIdentity();
					glDisable(GL_CULL_FACE);
					//glDisable(GL_TEXTURE_2D);
					glDisable(GL_BLEND);
					glBindTexture(GL_TEXTURE_2D, outputTex);
					glBegin(GL_QUADS);
					glColor3f(1, 0, 0);
					glTexCoord2f(0, 0); glVertex3f(-1,-1,0);
					glColor3f(0, 1, 0);
					glTexCoord2f(0, 1); glVertex3f(-1,1,0);
					glColor3f(0, 0, 1);
					glTexCoord2f(1, 1); glVertex3f(1,1,0);
					glColor3f(1, 1, 1);
					glTexCoord2f(1, 0); glVertex3f(1,-1,0);
					glEnd();
					Display.update();
					Thread.sleep(500);
				} catch(Exception e) {
					e.printStackTrace();
				}
			}
			
			glDeleteTextures(inputTex);
		}
	}
}
