package mods.immibis.infiview;

import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.lwjgl.opengl.ARBSync;
import org.lwjgl.opengl.GLSync;

/**
 * Manages the queue of texture downloads.
 */
public class TextureDownloadQueue {
	private static final int MAX_QUEUE_ITEMS_IN_FLIGHT = 1; // maximum number of requests we'll have dispatched but not completed at any given time
	
	private class QueueItem {
		QueueItem next;
		PendingSaveOperation opS;
		PendingMergeOperation opM;
		GLSync sync;
	}
	
	private QueueItem firstQueueItem = null;
	private QueueItem lastQueueItem = null;
	private int numQueueItemsInFlight = 0;
	
	private List<PendingSaveOperation> saveQueue = new ArrayList<>();
	private List<PendingMergeOperation> mergeQueue = new ArrayList<>();
	
	public synchronized void enqueue(PendingSaveOperation op) {
		saveQueue.add(op);
	}

	public synchronized void enqueue(PendingMergeOperation op) {
		mergeQueue.add(op);
	}
	
	public void tick(boolean waitForFinish) {
		while(firstQueueItem != null) {
			if(!waitForFinish) {
				// Check whether this download is ready.
				// It might seem counter-intuitive to have if(!shouldWait) {wait(...)}
				// but notice we're waiting with a timeout of zero.
				int waitResult = ARBSync.glClientWaitSync(firstQueueItem.sync, ARBSync.GL_SYNC_FLUSH_COMMANDS_BIT, 0);
				if(waitResult == ARBSync.GL_TIMEOUT_EXPIRED)
					break;
				if(waitResult != ARBSync.GL_ALREADY_SIGNALED)
					InfiViewMod.LOGGER.log(Level.WARN, "glClientWaitSync returned result code "+waitResult+". Assuming completed.");
			}
			
			ARBSync.glDeleteSync(firstQueueItem.sync);
			
			if(firstQueueItem.opS != null)
				firstQueueItem.opS.onDownloadComplete();
			else
				firstQueueItem.opM.onDownloadCompleted();
			
			assert numQueueItemsInFlight > 0;
			numQueueItemsInFlight--;
			
			firstQueueItem = firstQueueItem.next;
			if(firstQueueItem == null) {
				lastQueueItem = null;
			}
		}
		
		while(numQueueItemsInFlight < MAX_QUEUE_ITEMS_IN_FLIGHT) {
			PendingSaveOperation firstOpS = null;
			PendingMergeOperation firstOpM = null;
			synchronized(this) {
				if(saveQueue.size() == 0 && mergeQueue.size() == 0)
					break;
				else if(saveQueue.size() > 0)
					firstOpS = saveQueue.remove(saveQueue.size() - 1);
				else
					firstOpM = mergeQueue.remove(mergeQueue.size() - 1);
			}
			
			QueueItem qi = new QueueItem();
			qi.opS = firstOpS;
			qi.opM = firstOpM;
			if(firstOpS != null)
				firstOpS.doDownloadNow();
			else
				firstOpM.doDownloadNow();
			qi.sync = ARBSync.glFenceSync(ARBSync.GL_SYNC_GPU_COMMANDS_COMPLETE, 0);
			qi.next = null;
			if(firstQueueItem != null)
				lastQueueItem.next = qi;
			else
				firstQueueItem = qi;
			lastQueueItem = qi;
			
			numQueueItemsInFlight++;
		}
	}

	public boolean drain() {
		boolean didAnything = false;
		while(numQueueItemsInFlight != 0 || firstQueueItem != null) {
			tick(true);
			didAnything = true;
		}
		return didAnything;
	}
}
