package mods.immibis.infiview.capture;

import static org.lwjgl.opengl.ARBFramebufferObject.*;
import static org.lwjgl.opengl.GL11.*;

import java.nio.ByteBuffer;

import mods.immibis.infiview.InfiViewDirections;
import mods.immibis.infiview.InfiViewMod;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.OpenGlHelper;
import net.minecraft.client.renderer.WorldRenderer;
import net.minecraft.client.renderer.texture.TextureMap;

import org.apache.logging.log4j.Level;
import org.lwjgl.input.Keyboard;
import org.lwjgl.opengl.Display;
import org.lwjgl.opengl.GL12;
import org.lwjgl.opengl.GL14;

import static mods.immibis.infiview.InfiViewMod.*;

public class InfiViewCapturer {
	/**
	 * TODO call this something more concise, also move it to a utility class somewhere.
	 * 
	 * Returns the lowest number which is a power of two, and at least than the square root of the argument.
	 * Used to decide the size of a texture that can hold n images.
	 */
	static int sqrtRoundedUpToPowerOfTwo(int n) {
		for(int k = 0; k < 16; k++) {
			if((1 << (k*2)) >= n)
				return 1 << k;
		}
		throw new ArithmeticException("sqrtRoundedUpToPowerOfTwp("+n+")");
	}
	
	/**
	 * The result of capturing is a single texture, consisting
	 * of IMAGES_IN_ARRAY_BY_DIMENSION by IMAGES_IN_ARRAY_BY_DIMENSION
	 * sub-textures.
	 */
	public static final int IMAGES_IN_ARRAY_BY_DIMENSION = 8;
	static {assert IMAGES_IN_ARRAY_BY_DIMENSION == sqrtRoundedUpToPowerOfTwo(InfiViewDirections.NUM_DIRECTIONS);}
	public static final int ARRAY_TEXTURE_DIMENSION = IMAGES_IN_ARRAY_BY_DIMENSION * InfiViewMod.IMAGE_DIMENSION;
	public static final int ARRAY_TEXTURE_BYTES = ARRAY_TEXTURE_DIMENSION*ARRAY_TEXTURE_DIMENSION*4;
	
	private static int lastTextureSize = -1;
	private static int framebuffer = -1;
	private static int depthRenderbuffer = -1;
	
	private static void init(int textureSize) {
		if(framebuffer == -1) {
			framebuffer = glGenFramebuffers();
			depthRenderbuffer = glGenRenderbuffers();
		}
		
		if(lastTextureSize != textureSize) {
	    	glBindRenderbuffer(GL_RENDERBUFFER, depthRenderbuffer);
	    	glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT, textureSize, textureSize);
	    	
	    	glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
	    	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, depthRenderbuffer);
	    	
	    	lastTextureSize = textureSize;
	    } else {
	    	glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
	    }
    	
	}
    

	// ssaaLevel is the number of mipmap levels above the base level that will be downloaded.
	// This level will be downloadImageSize*downloadImageSize pixels (per image in the array).
	// Therefore, the size of the images actually rendered to will be (downloadImageSize << ssaaLevel).
	public static void renderAllDirections(WorldRenderer r, int downloadImageSize, int ssaaLevel, int texID) {
		
		int renderImageSize = downloadImageSize << ssaaLevel;
		int textureSize = renderImageSize * IMAGES_IN_ARRAY_BY_DIMENSION;
		init(textureSize);
		
		
		glBindTexture(GL_TEXTURE_2D, texID);
		glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_BASE_LEVEL, 0);
		glTexParameteri(GL_TEXTURE_2D, GL12.GL_TEXTURE_MAX_LEVEL, ssaaLevel);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, textureSize, textureSize, 0, GL_RGBA, GL_UNSIGNED_BYTE, (ByteBuffer)null);
		glBindFramebuffer(GL_FRAMEBUFFER, framebuffer);
		glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, texID, 0);
		glBindTexture(GL_TEXTURE_2D, 0);
		
		if(glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE) {
			InfiViewMod.LOGGER.log(Level.WARN, "InfiViewCapturer: Framebuffer error! glCheckFramebufferStatus returned "+glCheckFramebufferStatus(GL_FRAMEBUFFER));
			return;
		}
		
		glClearColor(0,0,0,0);
		
		// GL_LIGHTING_BIT for glShadeModel setting
		// GL_DEPTH_BUFFER_BIT for glDepthMask setting (which apparently was false)
		// GL_COLOR_BUFFER_BIT for glBlendFunc setting
		glPushAttrib(GL_ENABLE_BIT | GL_LIGHTING_BIT | GL_VIEWPORT_BIT | GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
        
		glEnable(GL_ALPHA_TEST);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_BLEND);
		glShadeModel(GL_SMOOTH);
		glEnable(GL_SCISSOR_TEST);
		glEnable(GL_CULL_FACE);
		
		// Input textures are not premultiplied, rendered output is premultiplied.
		// TODO review this
		GL14.glBlendFuncSeparate(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA, GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
		
		r.isInFrustum = true;
		for(int direction = 0; direction < InfiViewDirections.NUM_DIRECTIONS; direction++) {
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			double orthoSize = InfiViewDirections.getImageDimensionIngame(direction);
			glOrtho(-orthoSize, orthoSize, -orthoSize, orthoSize, -100000, 100000);
			glMatrixMode(GL_MODELVIEW);
			
			int imx = direction % IMAGES_IN_ARRAY_BY_DIMENSION;
			int imy = direction / IMAGES_IN_ARRAY_BY_DIMENSION;
			glViewport(renderImageSize*imx, renderImageSize*imy, renderImageSize, renderImageSize);
			glScissor(renderImageSize*imx, renderImageSize*imy, renderImageSize, renderImageSize);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			glLoadIdentity();
			InfiViewDirections.setGLViewAngleByDirection(direction);
			glTranslatef(-r.posXClip-8, -r.posYClip-8, -r.posZClip-8);
			Minecraft.getMinecraft().getTextureManager().bindTexture(TextureMap.locationBlocksTexture);
			
			glDepthMask(false);
			
			if(!r.skipRenderPass[0]) {
				// Render back-faces first, with depth only.
				// This is useful in the case of e.g. caves, where the outside of the cave will be INVISIBLE,
				// rather than showing through to the opposite side. This reduces file sizes by increasing the amount
				// of fully transparent pixels.
				glCullFace(GL_FRONT);
				glColorMask(false, false, false, false);
				glCallList(r.getGLCallListForPass(0));
				glColorMask(true, true, true, true);
			}
			
			glCullFace(GL_BACK);
		
			for(int pass = 0; pass < r.skipRenderPass.length; pass++) {
				if(r.skipRenderPass[pass])
					continue;
			
				glCallList(r.getGLCallListForPass(pass));
			}
		}
		
		glBindTexture(GL_TEXTURE_2D, texID);
		glEnable(GL_TEXTURE_2D); // ATI driver bug; glGenerateMipmaps requires GL_TEXTURE_2D enabled
		glGenerateMipmap(GL_TEXTURE_2D);
		
		glPopAttrib();
		
		glBindFramebuffer(GL_FRAMEBUFFER, 0);
		
		if(false) {
			glClearColor(1,0,0,1);
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			glMatrixMode(GL_PROJECTION);
			glDisable(GL_SCISSOR_TEST);
			glLoadIdentity();
			glOrtho(-1, 1, -1, 1, -1, 1);
			glMatrixMode(GL_MODELVIEW);
			glColor3f(1, 1, 1);
			glLoadIdentity();
			glDisable(GL_CULL_FACE);
			OpenGlHelper.setActiveTexture(OpenGlHelper.lightmapTexUnit);
			glDisable(GL_TEXTURE_2D);
			OpenGlHelper.setActiveTexture(OpenGlHelper.defaultTexUnit);
			glBegin(GL_QUADS);
			glTexCoord2f(0, 0); glVertex2f(-1, -1);
			glTexCoord2f(1, 0); glVertex2f( 1, -1);
			glTexCoord2f(1, 1); glVertex2f( 1,  1);
			glTexCoord2f(0, 1); glVertex2f(-1,  1);
			glEnd();
			Display.update();
			try {
				while(!Keyboard.isKeyDown(Keyboard.KEY_A)) {
					Thread.sleep(50);
					//Display.swapBuffers();
					Display.update();
					Display.update();
				}
			} catch (InterruptedException/* | LWJGLException*/ e) {
				e.printStackTrace();
			}
		}
	}



	public static void capture(WorldRenderer r, int tex) {
		InfiViewCapturer.renderAllDirections(r, IMAGE_DIMENSION, multiSampleLevel, tex);
	}
	

	// Returns true if all the pixels in this buffer have alpha 0.
	private static boolean isImageBufferEmpty(ByteBuffer buf) {
		buf.position(0).limit(buf.capacity());
		while(buf.hasRemaining()) {
			buf.get(); buf.get(); buf.get(); // skip RGB
			if(buf.get() != 0)
				return false;
		}
		return true;
	}
	
}
