package mods.immibis.infiview.storage;

import java.io.IOException;
import java.nio.ByteBuffer;

import org.apache.logging.log4j.Level;

import mods.immibis.infiview.InfiViewMod;
import mods.immibis.infiview.NodeImageData;
import mods.immibis.infiview.PendingMergeOperation;

/**
 * In-memory representation of a quad-tree node.
 */
public class CachedQuadTreeNode {
	private final StorageManager manager;	
	private final int nodeID;
	private final int size;
	CachedQuadTreeNode parent;
	CachedQuadTreeNode childNN, childNP, childPN, childPP;
	boolean areChildrenLoaded;
	public NodeImageData loadedImageData;
	CachedQuadTreeNode(int nodeID, StorageManager manager, int size, CachedQuadTreeNode parent) {
		this.nodeID = nodeID;
		this.manager = manager;
		this.size = size;
		this.parent = parent;
	}
	
	public int getNodeID() {
		return nodeID;
	}

	/**
	 * If this node has an image for the given minichunk number,
	 * then delete it, and mark parents as dirty.
	 */
	public void deleteImageAndMarkParentsForUpdate(int y) {
		assert Thread.currentThread() == InfiViewMod.mainThread;
		assert y >= 0 && y <= 15;
		int imagePtrPos = manager.quadTreeFile.getImageSeriesRefPoint(nodeID, y);
		final long imageBlockID = manager.quadTreeFile.getImageSeries(imagePtrPos);
		if(imageBlockID == -1)
			return;
		
		// Lock the block to prevent double-deletes (if the compactor deletes it before we delete it)
		manager.imageDataFile2.lockBlock(imageBlockID);
		
		int ptr = manager.quadTreeFile.getImageSeriesRefPoint(nodeID, y);
		synchronized(manager.imageDataFile2) { // Ensure ordering with respect to lock. Maybe not needed?
			manager.quadTreeFile.base.file.mapping.putLong(ptr, -1);
		}
		InfiViewMod.ioThread.enqueueTrivial(new Runnable() {
			@Override
			public void run() {
				// markBlockUnused can execute compaction, but only after marking the block as unused.
				manager.imageDataFile2.unlockBlock(imageBlockID);
				manager.imageDataFile2.markBlockUnused(imageBlockID);
			}
		});
		if(parent != null)
			parent.markForIntermediateNodeUpdate(y/2);
	}
	
	public void markForIntermediateNodeUpdate(int posY) {
		if(size == 1)
			throw new RuntimeException("Cannot do an intermediate node update on a leaf node.");
		
		loadChildren();
		
		mergeRebuildStateWord(RS_THIS_NEEDS_REBUILD, 1 << posY);
		setRebuildStateWord(getRebuildStateWord() & RS_THIS_NEEDS_REBUILD);
		CachedQuadTreeNode cur = parent;
		while(cur != null) {
			posY /= 2;
			cur.mergeRebuildStateWord(RS_CHILDREN_NEED_REBUILD, 1 << posY);
			cur = cur.parent;
		}
	}
	
	// Since intermediate nodes only have 8 minichunks at most, we reuse the image pointer for minichunk 15
	// to store the node's rebuild state word.
	// Bits 0 through 7 are one of the following constants. (Any other value is treated as RS_CHILDREN_NEED_REBUILD.
	// In particular, it's initialized to 0xFF or 0x00).
	// Bits 8 through 15 are a mask of which minichunks need to be rebuilt.
	private static final int RS_OKAY = 0x40;
	private static final int RS_THIS_NEEDS_REBUILD = 0x80;
	private static final int RS_CHILDREN_NEED_REBUILD = 0xC0;
	private static final int RS_STATE_MASK = 0xFF;
	private int getRebuildStateWord() {
		if(size == 1)
			return RS_OKAY;
		return (int)manager.quadTreeFile.getImageSeries(nodeID, 15);
	}
	private void setRebuildStateWord(int state) {
		if(size == 1)
			throw new RuntimeException("Leaf nodes don't have a rebuild state.");
		manager.quadTreeFile.base.file.mapping.putLong(getImageSeriesRefPoint(15), (long)state);
	}
	private void mergeRebuildStateWord(int newState, int newMaskBits) {
		assert (newMaskBits & ~0xFF) == 0;
		assert (newState & ~0xFF) == 0;
		assert (newMaskBits & (0xFF << getNumMinichunks())) == 0; // We're not marking any out-of-range minichunks.
			
		int old = getRebuildStateWord();
		if((old & RS_STATE_MASK) < newState)
			old = (old & ~RS_STATE_MASK) | newState;
		old |= newMaskBits << 8;
		setRebuildStateWord(old);
	}

	public int getSize() {
		return size;
	}

	public CachedQuadTreeNode getChild(int childNumber) {
		if(!areChildrenLoaded)
			loadChildren();
		switch(childNumber) {
		case 0: return childNN;
		case 1: return childPN;
		case 2: return childNP;
		case 3: return childPP;
		default: throw new IllegalArgumentException("childNumber "+childNumber);
		}
	}

	void loadChildren() {
		if(size > 1) {
			UnboundedQuadTreeFileL1 l1 = manager.quadTreeFile.base;
			int pos = l1.getNodePosition(nodeID);
			childNN = createChildNode(l1.file.mapping.getInt(pos));
			childPN = createChildNode(l1.file.mapping.getInt(pos+4));
			childNP = createChildNode(l1.file.mapping.getInt(pos+8));
			childPP = createChildNode(l1.file.mapping.getInt(pos+12));
		}
		areChildrenLoaded = true;
	}

	private CachedQuadTreeNode createChildNode(int nodeID) {
		if(nodeID == -1)
			return null;
		assert size > 1;
		return new CachedQuadTreeNode(nodeID, manager, size>>1, this);
	}

	CachedQuadTreeNode getOrCreateChild(int childNumber) throws IOException {
		CachedQuadTreeNode c = getChild(childNumber);
		if(c == null) {
			int childNodeID = manager.quadTreeFile.allocateNewChild(nodeID, childNumber);
			assert childNodeID != -1;
			c = createChildNode(childNodeID);
			switch(childNumber) {
			case 0: childNN = c; break;
			case 1: childPN = c; break;
			case 2: childNP = c; break;
			case 3: childPP = c; break;
			default: throw new AssertionError("childNumber "+childNumber);
			}
		}
		return c;
	}

	public CachedQuadTreeNode getParent() {
		return parent;
	}

	public int getImageSeriesRefPoint(int y) {
		return manager.quadTreeFile.getImageSeriesRefPoint(nodeID, y);
	}
	
	private int getNumMinichunks() {
		return Math.max(16/size, 1);
	}

	private short mergingMask = 0; // Bit is set for each minichunk that is being rebuilt. Only applies to non-leaf nodes.
	public void queueMergeIfNeeded() {
		if(size == 1)
			return;
		int rsWord = getRebuildStateWord();
		int rs = rsWord & RS_STATE_MASK;
		if(rs == RS_OKAY)
			return;
		else if(rs == RS_THIS_NEEDS_REBUILD) {
			int rsMask = (rsWord >> 8) & 0xFF;
			rsMask &= (1 << getNumMinichunks()) - 1;
			if((~mergingMask & rsMask) != 0) {
				for(int k = 0; k < 8; k++) {
					if((~mergingMask & rsMask & (1 << k)) != 0) {
						mergingMask |= 1 << k;
						InfiViewMod.mergeQueue.add(new PendingMergeOperation(this, k));
					}
				}
			}
			if(rsMask == 0) {
				// Done rebuilding.
				setRebuildStateWord((rsWord & ~RS_STATE_MASK) | RS_OKAY);
			}
		} else {
			// Treat any other value as RS_CHILDREN_NEED_REBUILD
			// (Including RS_CHILDREN_NEED_REBUILD, as well as unknown values)
			loadChildren();
			if(childNN != null && (childNN.getRebuildStateWord() & RS_STATE_MASK) != RS_OKAY)
				childNN.queueMergeIfNeeded();
			else if(childNP != null && (childNP.getRebuildStateWord() & RS_STATE_MASK) != RS_OKAY)
				childNP.queueMergeIfNeeded();
			else if(childPN != null && (childPN.getRebuildStateWord() & RS_STATE_MASK) != RS_OKAY)
				childPN.queueMergeIfNeeded();
			else if(childPP != null && (childPP.getRebuildStateWord() & RS_STATE_MASK) != RS_OKAY)
				childPP.queueMergeIfNeeded();
			else {
				// All children are fully rebuilt.
				setRebuildStateWord((rsWord & ~RS_STATE_MASK) | RS_THIS_NEEDS_REBUILD);
			}
		}
	}

	public void writeCompressedImageSeriesNow(int posY, byte[] compressedImage) {
		// This call does not access the disk and is safe to call on any thread.
		final int imageSeriesRefPoint = manager.quadTreeFile.getImageSeriesRefPoint(nodeID, posY);
		
		final long newBlockNum;
		try {
			newBlockNum = manager.imageDataFile2.allocateBlock(compressedImage.length, imageSeriesRefPoint);
		} catch(IOException e) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Failed to allocate a block of "+compressedImage.length+" bytes", e);
			return;
		}
		
		final long blockPos = manager.imageDataFile2.getFilePosition(newBlockNum);
		try {
			InfiViewMod.LOGGER.log(Level.TRACE, "Writing image data at "+blockPos);
			manager.imageDataFile2.write(ByteBuffer.wrap(compressedImage), blockPos);
		} catch (IOException e) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Failed to write "+compressedImage.length+" bytes to block "+blockPos, e);
			return;
		}
		
		// Prevent the compactor from deleting our block before we've made it active (by setting the pointer in the quadtree file)
		// Note that the compactor only runs on this thread, so as long as we're running it isn't, so there isn't a
		// race condition where the block could be deleted before this line executes.
		manager.imageDataFile2.lockBlock(newBlockNum);
		
		InfiViewMod.mainThreadCallbackQueue.add(new Runnable() {
			@Override
			public void run() {
				final long oldBlockNum = manager.quadTreeFile.getImageSeries(imageSeriesRefPoint);
				if(oldBlockNum != -1 && oldBlockNum != newBlockNum) { // second part should always be true
					// Prevent the compactor from deleting our block before we've deleted it (to prevent double-deletes)
					manager.imageDataFile2.lockBlock(oldBlockNum);
					InfiViewMod.ioThread.enqueueTrivial(new Runnable() {
						@Override
						public void run() {
							// markBlockUnused can execute compaction, but only after it marks the block as unused.
							manager.imageDataFile2.unlockBlock(oldBlockNum);
							manager.imageDataFile2.markBlockUnused(oldBlockNum);
						}
					});
				}
				synchronized(manager.imageDataFile2) { // Ensure ordering with respect to lock/unlock. Maybe not needed?
					manager.quadTreeFile.base.file.mapping.putLong(imageSeriesRefPoint, newBlockNum);
				}
				manager.imageDataFile2.unlockBlock(newBlockNum);
			}
		});
	}

	/**
	 * Called from the /infiview_reset command only to reset "stuck" state.
	 */
	public void reset(int flags) {
		mergingMask = 0;
	}

	public void onDoneMerging(int y) {
		mergingMask &= ~(1 << y);
		setRebuildStateWord(getRebuildStateWord() & ~(1 << (y + 8)));
	}
}
