package mods.immibis.infiview.storage;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.FileChannel.MapMode;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

public final class ExpandableMemoryMappedFile {
	private FileChannel fc;
	private FileLock lock;
	
	public MappedByteBuffer mapping;
		
	public ExpandableMemoryMappedFile(File f) throws IOException {
		fc = FileChannel.open(Paths.get(f.toURI()), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.READ);
		lock = fc.lock();
		
		mapBuffer();
	}
	
	public void close() {
		try {
			lock.release();
		} catch (IOException e) {
			e.printStackTrace();
		}
		try {
			fc.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		fc = null;
	}
	
	private void mapBuffer() throws IOException {
		if(fc.size() == 0)
			mapping = null;
		else {
			mapping = fc.map(MapMode.READ_WRITE, 0, fc.size());
			mapping.order(ByteOrder.nativeOrder());
		}
	}

	private static final ByteBuffer EMPTY_BUFFER = ByteBuffer.allocateDirect(4096);
	/** Returns true if the file was expanded, false if it was already at least that big. */
	public boolean expandFileTo(long l) throws IOException {
		long pos = fc.size();
		long diff = l - pos;
		if(diff <= 0)
			return false;
		synchronized(EMPTY_BUFFER) {
			while(diff >= 4096) {
				EMPTY_BUFFER.position(0).limit(4096);
				fc.write(EMPTY_BUFFER, pos);
				diff -= 4096;
				pos += 4096;
			}
			if(diff > 0) {
				EMPTY_BUFFER.position(0).limit((int)diff);
				fc.write(EMPTY_BUFFER, pos);
			}
		}
		mapBuffer();
		return true;
	}
}
