package mods.immibis.infiview.storage;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import mods.immibis.infiview.InfiViewMod;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * The file is divided into 2MB megachunks.
 * For each megachunks there's a megachunk header containing the next free position in the chunk,
 * then a bunch of blocks.
 * Each block has a header containing the block's size, and the location in the index file that stores a
 * pointer to the block (so it can be updated during compaction), and the amount of space allocated for the block
 * (can be > block size, and then shrinks during compaction). Both the block size and the allocated size include the header.
 * 
 * Block numbers are opaque numbers (which are actually pointers to the block header).
 * They are stored somewhere in the index file. We store the location where the block number is stored,
 * so that if the block is moved in the file (which changes its block number) we can 
 * 
 * This is NOT multithreading safe (because of headerBuf). All calls to this should happen on the I/O thread.
 * 
 * TODO: EVERYTHING INVOLVING INDEXFILE is thread-unsafe! (the index file should only be accessed on the Minecraft client thread)
 */
public class ImageDataFile2 {
	
	// 4-byte size, 4-byte reverse pointer, 4-byte allocated size (>= size), 20 reserved. Sizes include header.
	private static final int BLOCK_HEADER_BYTES = 32;

	// 4-byte used amount, 28 reserved
	private static final int MEGACHUNK_HEADER_BYTES = 32;

	private static final int MEGACHUNK_BYTES = 1 << 21; // 2MB, including header
	
	final FileChannel fc;
	public final ExpandableMemoryMappedFile indexFile;
	final Thread thread;
	
	private static final Logger COMPACTION_LOGGER = LogManager.getLogger("InfiView-Compactor");
	// Always logs at level INFO or above, since that's default-enabled.
	private static final Logger VERIFIER_LOGGER = LogManager.getLogger("InfiView-Verifier");
	private static final boolean VERIFIER_ENABLED = Boolean.getBoolean("immibis.infiview.verifyDataFileIntegrity");
	
	// Megachunk number -> lock count
	// Blocks in locked megachunks will not be moved (until the megachunk is unlocked).
	private int[] lockedMegachunks;
	
	private synchronized void lockMegachunk(int mcNum) {
		lockedMegachunks[mcNum]++;
	}
	
	private synchronized void unlockMegachunk(int mcNum) {
		lockedMegachunks[mcNum]--;
		assert lockedMegachunks[mcNum] >= 0;
	}
	
	public void lockBlock(long blockNum) {
		lockMegachunk((int)(blockNum / MEGACHUNK_BYTES));
	}
	
	public void unlockBlock(long blockNum) {
		unlockMegachunk((int)(blockNum / MEGACHUNK_BYTES));
	}
	
	public ImageDataFile2(File f, ExpandableMemoryMappedFile indexFile) throws IOException {
		this.fc = FileChannel.open(f.toPath(), StandardOpenOption.WRITE, StandardOpenOption.READ, StandardOpenOption.CREATE);
		this.indexFile = indexFile;
		
		this.thread = Thread.currentThread();
		
		long curSize = fc.size();
		
		assert (curSize % MEGACHUNK_BYTES) == 0;
		numMegachunks = (int)(curSize / MEGACHUNK_BYTES);
		
		allocatedEndPerMegachunk = new int[numMegachunks * 2];
		usedBytesPerMegachunk = new int[allocatedEndPerMegachunk.length];
		lockedMegachunks = new int[allocatedEndPerMegachunk.length];
		
		for(int k = 0; k < numMegachunks; k++) {
			headerBuf.position(0).limit(4);
			fc.read(headerBuf, k * (long)MEGACHUNK_BYTES);
			allocatedEndPerMegachunk[k] = headerBuf.getInt(0);
			try {
				usedBytesPerMegachunk[k] = countActuallyUsedBytesInMegachunk(k);
			} catch(IOException e) {
				new Exception("Megachunk "+k+"appears corrupted; resetting it").printStackTrace();
				allocatedEndPerMegachunk[k] = usedBytesPerMegachunk[k] = MEGACHUNK_HEADER_BYTES;
			}
			
			if(usedBytesPerMegachunk[k] < allocatedEndPerMegachunk[k] / 2 && allocatedEndPerMegachunk[k] > 262144)
				compactMegachunk(k);
		}
		
		verifyConsistency();
	}
	
	public void read(ByteBuffer buf, long pos) throws IOException {
		fc.position(pos);
		while(buf.hasRemaining())
			if(fc.read(buf) <= 0)
				throw new IOException("Unexpected EOF");
	}
	
	public void write(ByteBuffer buf, long pos) throws IOException {
		fc.position(pos);
		while(buf.hasRemaining())
			fc.write(buf);
	}
	
	private int countActuallyUsedBytesInMegachunk(int id) throws IOException {
		long pos = id * (long)MEGACHUNK_BYTES;
		long end = pos + allocatedEndPerMegachunk[id];
		pos += MEGACHUNK_HEADER_BYTES;
		
		int total = MEGACHUNK_HEADER_BYTES;
		
		while(pos < end) {
			headerBuf.position(0).limit(BLOCK_HEADER_BYTES);
			read(headerBuf, pos);
			total += headerBuf.getInt(0);
			pos += headerBuf.getInt(8);
		}
		
		return total;
	}

	private void compactMegachunk(int id) throws IOException {
		assert Thread.currentThread() == thread;
		
		long megachunkStart = id * (long)MEGACHUNK_BYTES;
		
		// End of uncompacted space, relative to the megachunk
		int end = allocatedEndPerMegachunk[id];
		// Start of uncompacted space, relative to the megachunk
		int pos = MEGACHUNK_HEADER_BYTES;
		
		// End of compacted space, relative to the megachunk
		int moveTo = MEGACHUNK_HEADER_BYTES;
		
		COMPACTION_LOGGER.log(Level.TRACE, "Compacting megachunk {}", id);
		
		while(pos < end)
		{
			long blockNum = megachunkStart + pos;
			
			// Read next block header
			headerBuf.position(0).limit(BLOCK_HEADER_BYTES);
			read(headerBuf, blockNum);
			int usedSize = headerBuf.getInt(0); // including header
			int reversePointer = headerBuf.getInt(4);
			int allocatedSize = headerBuf.getInt(8); // including header
			
			if(indexFile.mapping.getLong(reversePointer) != blockNum) {
				COMPACTION_LOGGER.log(Level.TRACE, "Removing orphaned chunk at {}", blockNum);
				pos += allocatedSize;
				continue;
			}
			
			indexFile.mapping.putLong(reversePointer, megachunkStart + moveTo);
			
			copy(megachunkStart+pos, megachunkStart+moveTo, usedSize);
			
			headerBuf.putInt(8, usedSize); // usedSize = allocatedSize, because we're compacting the empty space.
			
			// Update block header: set allocatedSize to usedSize, since we compact the unused space.
			headerBuf.position(0).limit(BLOCK_HEADER_BYTES);
			write(headerBuf, megachunkStart+moveTo);
			
			pos += allocatedSize;
			moveTo += usedSize;
		}
		
		zeroFill(megachunkStart+moveTo, megachunkStart+MEGACHUNK_BYTES);
		
		// Update megachunk header: set freeDataStart to moveTo.
		headerBuf.position(0).limit(MEGACHUNK_HEADER_BYTES);
		read(headerBuf, megachunkStart);
		
		headerBuf.putInt(0, moveTo);
		allocatedEndPerMegachunk[id] = moveTo;
		usedBytesPerMegachunk[id] = moveTo;
		
		headerBuf.position(0).limit(MEGACHUNK_HEADER_BYTES);
		write(headerBuf, megachunkStart);
		
		verifyMegachunkConsistency(id);
	}
	
	private void zeroFill(long pos, long end) throws IOException {
		assert Thread.currentThread() == thread;
		
		int chunkSize = largeZeroBuf.capacity();
		fc.position(pos);
		while(end - pos > chunkSize) {
			largeZeroBuf.position(0).limit(chunkSize);
			pos += fc.write(largeZeroBuf);
		}
		while(end > pos) {
			largeZeroBuf.position(0).limit((int)(end - pos));
			pos += fc.write(largeZeroBuf);
		}
	}
	
	private void copy(long from, long to, long size) throws IOException {
		assert Thread.currentThread() == thread;
		
		int chunkSize = largeBuf.capacity();
		while(size > 0) {
			largeBuf.position(0).limit((int)Math.min(size, chunkSize));
			int numRead = fc.read(largeBuf, from);
			if(numRead <= 0)
				throw new IOException("Unexpected EOF");
			largeBuf.flip();
			while(largeBuf.hasRemaining()) {
				int numWritten = fc.write(largeBuf, to);
				from += numWritten;
				to += numWritten;
				size -= numWritten;
			}
		}
	}

	private final ByteBuffer largeBuf = ByteBuffer.allocateDirect(65536);
	private final ByteBuffer largeZeroBuf = ByteBuffer.allocateDirect(65536);
	private final ByteBuffer headerBuf = ByteBuffer.allocateDirect(BLOCK_HEADER_BYTES);
	
	private int[] allocatedEndPerMegachunk;
	private int[] usedBytesPerMegachunk;
	private int numMegachunks = 0;
	
	private void allocateMegachunk() throws IOException {
		assert Thread.currentThread() == this.thread;
		
		long curSize = fc.size();
		assert (curSize % MEGACHUNK_BYTES) == 0;
		numMegachunks = (int)(curSize / MEGACHUNK_BYTES);
		
		if(allocatedEndPerMegachunk.length == numMegachunks) {
			allocatedEndPerMegachunk = Arrays.copyOf(allocatedEndPerMegachunk, (numMegachunks+1)*2);
			usedBytesPerMegachunk = Arrays.copyOf(usedBytesPerMegachunk, allocatedEndPerMegachunk.length);
			lockedMegachunks = Arrays.copyOf(lockedMegachunks, allocatedEndPerMegachunk.length);
		}
		
		assert (MEGACHUNK_BYTES % largeZeroBuf.capacity()) == 0;
		fc.position(curSize);
		for(int k = 0; k < MEGACHUNK_BYTES; k += largeZeroBuf.capacity()) {
			largeZeroBuf.position(0).limit(largeZeroBuf.capacity());
			fc.write(largeZeroBuf);
		}
		
		headerBuf.position(0).limit(4);
		headerBuf.putInt(0, MEGACHUNK_HEADER_BYTES);
		write(headerBuf, numMegachunks*(long)MEGACHUNK_BYTES);
		allocatedEndPerMegachunk[numMegachunks] = MEGACHUNK_HEADER_BYTES;
		usedBytesPerMegachunk[numMegachunks] = MEGACHUNK_HEADER_BYTES;
		
		numMegachunks++;
	}
	
	private long allocateBlock(int size, int pointerLocation, int megachunkIndex) throws IOException {
		assert Thread.currentThread() == this.thread;
		
		if(allocatedEndPerMegachunk[megachunkIndex] < MEGACHUNK_BYTES - MEGACHUNK_HEADER_BYTES - size) {
			int posInChunk = allocatedEndPerMegachunk[megachunkIndex];
			allocatedEndPerMegachunk[megachunkIndex] += size;
			
			long blocknum = megachunkIndex*(long)MEGACHUNK_BYTES + posInChunk;
			
			// Write block header
			headerBuf.position(0).limit(BLOCK_HEADER_BYTES);
			for(int i = 0; i < BLOCK_HEADER_BYTES; i += 4)
				headerBuf.putInt(i, 0);
			headerBuf.putInt(0, size);
			headerBuf.putInt(4, pointerLocation);
			headerBuf.putInt(8, size);
			fc.write(headerBuf, blocknum);
			
			// Update megachunk header
			headerBuf.position(0).limit(4);
			headerBuf.putInt(0, allocatedEndPerMegachunk[megachunkIndex]);
			fc.write(headerBuf, megachunkIndex*(long)MEGACHUNK_BYTES);
			
			usedBytesPerMegachunk[megachunkIndex] += size;
			
			return blocknum;
		}
		return -1;
	}
	
	/**
	 * Allocates a block.
	 * @param size The size of the block.
	 * @param pointerLocation The location in the index file where the block number will be stored.
	 * @return The block number of the new block.
	 * @throws IOException
	 */
	public long allocateBlock(int size, int pointerLocation) throws IOException {
		assert Thread.currentThread() == this.thread;
		
		size += BLOCK_HEADER_BYTES;
		for(int k = 0; k < numMegachunks; k++) {
			long result = allocateBlock(size, pointerLocation, k);
			if(result != -1) {
				verifyConsistency();
				return result;
			}
		}
		
		assert size <= MEGACHUNK_BYTES - MEGACHUNK_HEADER_BYTES;
		
		allocateMegachunk();
		long result = allocateBlock(size, pointerLocation, numMegachunks-1);
		assert result != -1;
		verifyConsistency();
		return result;
	}
	
	/**
	 * Gets the logical size of a block.
	 * Returns 0 on error.
	 * @param blockNum The block number.
	 */
	public int getBlockSize(long blockNum) {
		assert Thread.currentThread() == this.thread;
		
		try {
			headerBuf.position(0).limit(4);
			fc.position(blockNum);
			while(headerBuf.hasRemaining())
				if(fc.read(headerBuf) <= 0)
					throw new IOException("Unexpected EOF");
			return headerBuf.getInt(0) - BLOCK_HEADER_BYTES;
		} catch(IOException e) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Failed to get size of block "+blockNum, e);
			return 0;
		}
	}
	
	/**
	 * Gets the capacity of a block.
	 * @param blockNum The block number.
	 * @throws IOException 
	 */
	public int getBlockCapacity(long blockNum) throws IOException {
		assert Thread.currentThread() == this.thread;
		
		headerBuf.position(0).limit(4);
		fc.position(blockNum + 8);
		while(headerBuf.hasRemaining())
			if(fc.read(headerBuf) <= 0)
				throw new IOException("Unexpected EOF");
		return headerBuf.getInt(0) - BLOCK_HEADER_BYTES;
	}
	
	/**
	 * Returns the position in the data file where the block's payload starts.
	 * @param blockNum The block number.
	 */ 
	public long getFilePosition(long blocknum) {
		return blocknum + BLOCK_HEADER_BYTES;
	}

	/**
	 * Sets the used size, and reverse pointer, of a block.
	 * @throws IOException 
	 */
	public void reuseBlock(long blockNum, int size, int reversePointer) throws IOException {
		assert Thread.currentThread() == this.thread;
		
		verifyConsistency();
		
		int oldSize = getBlockSize(blockNum) + BLOCK_HEADER_BYTES;
		
		size += BLOCK_HEADER_BYTES;
		
		headerBuf.position(0).limit(8);
		headerBuf.putInt(0, size);
		headerBuf.putInt(4, reversePointer);
		fc.position(blockNum);
		while(headerBuf.hasRemaining())
			fc.write(headerBuf);

		int mc = (int)(blockNum / MEGACHUNK_BYTES);
		usedBytesPerMegachunk[mc] += size - oldSize;
		
		verifyConsistency();
	}

	public void markBlockUnused(long blockNum) {
		assert Thread.currentThread() == this.thread;
		
		int mc = (int)(blockNum / MEGACHUNK_BYTES);
		
		try {
			reuseBlock(blockNum, BLOCK_HEADER_BYTES, 0);

			verifyConsistency();
			
			if(usedBytesPerMegachunk[mc] < allocatedEndPerMegachunk[mc] / 2 && allocatedEndPerMegachunk[mc] > 262144) {
				int lockCount;
				synchronized(this) {
					lockCount = lockedMegachunks[mc];
				}
				assert lockCount >= 0;
				if(lockCount == 0)
					compactMegachunk(mc);
			}
		} catch(IOException e) {
			InfiViewMod.LOGGER.log(Level.ERROR, "Failed to mark block "+blockNum+" as unused.", e);
		}
	}

	private void verifyConsistency() throws IOException {
		if(!VERIFIER_ENABLED) return;
		
		try {
			assert numMegachunks * (long)MEGACHUNK_BYTES == fc.size();
			for(int k = 0; k < numMegachunks; k++) {
				verifyMegachunkConsistency(k);
			}
		} catch(AssertionError | IOException  e) {
			logConsistencyError(e);
		}
	}

	private void logConsistencyError(Throwable e) {
		if(e instanceof IOException) {
			VERIFIER_LOGGER.log(Level.WARN, "Encountered an I/O error while verifying", e);
			System.exit(1);
		} else if(e instanceof AssertionError) {
			if(!e.getStackTrace()[0].getClassName().equals(ImageDataFile2.class.getName()))
				throw (AssertionError)e; // not from the verifier!
			VERIFIER_LOGGER.log(Level.WARN, "Integrity verification failed", e);
			System.exit(1);
		} else {
			new AssertionError("Expected IOException or AssertionError, got "+e, e).printStackTrace();
		}
	}

	private void verifyMegachunkConsistency(int id) throws IOException {
		if(!VERIFIER_ENABLED) return;
		
		try {
			long mcStart = id * (long)MEGACHUNK_BYTES;
			
			headerBuf.position(0).limit(MEGACHUNK_HEADER_BYTES);
			read(headerBuf, mcStart);
			int mcAllocEnd = headerBuf.getInt(0);
			assert mcAllocEnd <= MEGACHUNK_BYTES;
			assert mcAllocEnd >= MEGACHUNK_HEADER_BYTES;
			
			int totalUsed = MEGACHUNK_HEADER_BYTES;
			
			long pos = mcStart + MEGACHUNK_HEADER_BYTES;
			while(pos < mcStart + mcAllocEnd) {
				headerBuf.position(0).limit(BLOCK_HEADER_BYTES);
				read(headerBuf, pos);
				int usedSize = headerBuf.getInt(0);
				int reversePtr = headerBuf.getInt(4);
				int allocSize = headerBuf.getInt(8);
				
				try {
					assert usedSize <= allocSize;
					assert usedSize >= BLOCK_HEADER_BYTES;
					assert pos + allocSize <= mcStart + mcAllocEnd;
					assert reversePtr >= 0;
				} catch(AssertionError e) {
					logConsistencyError(e);
				}
				
				totalUsed += usedSize;
				pos += allocSize;
			}
			assert pos == mcStart + mcAllocEnd;
			
			int usedDiff = usedBytesPerMegachunk[id] - totalUsed;
			if(usedDiff != 0) {
				VERIFIER_LOGGER.log(Level.WARN, "Megachunk "+id+" contains "+totalUsed+" used bytes, but we thought it had "+usedBytesPerMegachunk[id], new Exception("Stack trace"));
				usedBytesPerMegachunk[id] = totalUsed;
			}
		} catch(AssertionError | IOException e) {
			logConsistencyError(e);
		}
	}

	public void close() {
		try {
			fc.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		// do not close indexFile
	}
}
