package mods.immibis.infiview.storage;

import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.plugins.jpeg.JPEGImageWriteParam;

import mods.immibis.infiview.InfiViewMod;

import org.apache.logging.log4j.Level;

public class ImageJPEGCompressor {

	// Compresses the 32bpp RGBA image in imageBuffer (starting at position 0 and going to the limit), returning the result.
	public static byte[] compressImage(ByteBuffer imageBuffer, int width, int height) {
		
		int npixels = imageBuffer.limit() / 4;
		int[] argbArray = new int[npixels];
		for(int k = 0; k < npixels; k++) {
			int i = k*4;
			// Convert RGBA as bytes to ARGB as packed int
			argbArray[k] = ((imageBuffer.get(i) & 255) << 16)
				| ((imageBuffer.get(i+1) & 255) << 8)
				| (imageBuffer.get(i+2) & 255)
				| (imageBuffer.get(i+3) << 24);
		}
		
		BufferedImage bufferedImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
		bufferedImage.setRGB(0, 0, width, height, argbArray, 0, width);
		ImageTypeSpecifier bufferedImageTypeSpec = ImageTypeSpecifier.createFromRenderedImage(bufferedImage);
		
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		
		// XXX: assumes there's at least one writer available
		ImageWriter writer = ImageIO.getImageWriters(bufferedImageTypeSpec, "jpeg").next();
		
		JPEGImageWriteParam writeParam = new JPEGImageWriteParam(null);
		writeParam.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
		writeParam.setCompressionQuality(0.3f);
			
		baos.reset();
		try {
			writer.setOutput(ImageIO.createImageOutputStream(baos));
			writer.write(null, new IIOImage(bufferedImage, null, null), writeParam);
		} catch(IOException e) {
			throw new RuntimeException(e);
		}
			
		InfiViewMod.LOGGER.log(Level.TRACE, "Compressed image is {} bytes", baos.size());
		return baos.toByteArray();
	}

	/**
	 * Decompresses the JPEG data at imageBuffer.position() through imageBuffer.limit()
	 * and stores the uncompressed RGBA data back in imageBuffer, starting at position 0.
	 */
	public static void decompressImage(ByteBuffer imageBuffer) throws IOException {
		
		BufferedImage bufferedImage = ImageIO.read(new ByteBufferInputStream(imageBuffer));
		
		int npixels = bufferedImage.getWidth() * bufferedImage.getHeight();
		assert npixels * 4 <= imageBuffer.capacity();
		
		int[] argbArray = new int[npixels];
		bufferedImage.getRGB(0, 0, bufferedImage.getWidth(), bufferedImage.getHeight(), argbArray, 0, bufferedImage.getWidth());
		imageBuffer.position(0).limit(npixels*4);
		for(int argb : argbArray) {
			// Convert ARGB as int to RGBA as bytes
			imageBuffer.put((byte)(argb >> 16));
			imageBuffer.put((byte)(argb >> 8));
			imageBuffer.put((byte)argb);
			imageBuffer.put((byte)(argb >> 24));
		}
	}

}
