package mods.immibis.infiview.storage;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.Semaphore;

import mods.immibis.infiview.InfiViewMod;

public final class StorageManager {
	public StorageManager(final File mcDir, final String name) throws IOException {
		rawQuadTreeFile = new ExpandableMemoryMappedFile(new File(mcDir, "infiview-"+name+".1.bin"));
		quadTreeFile = new UnboundedQuadTreeFileL2(rawQuadTreeFile);
		InfiViewMod.ioThread.enqueueTrivial(new Runnable() {
			@Override
			public void run() {
				try {
					imageDataFile2 = new ImageDataFile2(new File(mcDir, "infiview-"+name+".2.bin"), rawQuadTreeFile);
				} catch (IOException e) {
					throw new RuntimeException(e);
				}
			}
		});
		
		for(int k = 0; k < 4; k++)
			quadTreeFile.base.ensureRootNodeCreated(k);
		
		rootNN = new CachedQuadTreeNode(quadTreeFile.getRootQuadrantNodeID(0), this, 1 << quadTreeFile.getRootQuadrantScale(0), null);
		rootPN = new CachedQuadTreeNode(quadTreeFile.getRootQuadrantNodeID(1), this, 1 << quadTreeFile.getRootQuadrantScale(1), null);
		rootNP = new CachedQuadTreeNode(quadTreeFile.getRootQuadrantNodeID(2), this, 1 << quadTreeFile.getRootQuadrantScale(2), null);
		rootPP = new CachedQuadTreeNode(quadTreeFile.getRootQuadrantNodeID(3), this, 1 << quadTreeFile.getRootQuadrantScale(3), null);
	}
	
	private CachedQuadTreeNode rootNN, rootNP, rootPN, rootPP;
	
	private ExpandableMemoryMappedFile rawQuadTreeFile;
	// TODO make private
	public UnboundedQuadTreeFileL2 quadTreeFile;
	// TODO make private
	public ImageDataFile2 imageDataFile2;
	
	/**
	 * Visits nodes in approximate reverse order of distance to the 1x1 node at (refX, refY).
	 * (That is, the farthest nodes will be visited approximately first)
	 * 
	 * More specifically, this guarantees not to visit a node after visiting any node that
	 * is in the same direction and is closer on both the X and Z axes.
	 */
	public void visitQuadTreeBackToFront(int refX, int refZ, QuadTreeVisitor quadTreeVisitor) {
		
		boolean renderPXFirst = refX < 0;
		boolean renderPZFirst = refZ < 0;
		
		int scale;
		// TODO: is there a better way to order these calls with less duplication?
		if(renderPZFirst) {
			if(renderPXFirst) {
				scale = quadTreeFile.getRootQuadrantScale(3);
				visitQuadTreeBackToFront(refX, refZ, rootPP, 0, 0, scale, quadTreeVisitor);
			}
			scale = quadTreeFile.getRootQuadrantScale(2);
			visitQuadTreeBackToFront(refX, refZ, rootNP, -(1 << scale), 0, scale, quadTreeVisitor);
			if(!renderPXFirst) {
				scale = quadTreeFile.getRootQuadrantScale(3);
				visitQuadTreeBackToFront(refX, refZ, rootPP, 0, 0, scale, quadTreeVisitor);
			}
		}
		if(renderPXFirst) {
			scale = quadTreeFile.getRootQuadrantScale(1);
			visitQuadTreeBackToFront(refX, refZ, rootPN, 0, -(1 << scale), scale, quadTreeVisitor);
		}
		scale = quadTreeFile.getRootQuadrantScale(0);
		visitQuadTreeBackToFront(refX, refZ, rootNN, -(1 << scale), -(1 << scale), scale, quadTreeVisitor);
		if(!renderPXFirst) {
			scale = quadTreeFile.getRootQuadrantScale(1);
			visitQuadTreeBackToFront(refX, refZ, rootPN, 0, -(1 << scale), scale, quadTreeVisitor);
		}
		if(!renderPZFirst) {
			if(renderPXFirst) {
				scale = quadTreeFile.getRootQuadrantScale(3);
				visitQuadTreeBackToFront(refX, refZ, rootPP, 0, 0, scale, quadTreeVisitor);
			}
			scale = quadTreeFile.getRootQuadrantScale(2);
			visitQuadTreeBackToFront(refX, refZ, rootNP, -(1 << scale), 0, scale, quadTreeVisitor);
			if(!renderPXFirst) {
				scale = quadTreeFile.getRootQuadrantScale(3);
				visitQuadTreeBackToFront(refX, refZ, rootPP, 0, 0, scale, quadTreeVisitor);
			}
		}
	}

	private void visitQuadTreeBackToFront(int refX, int refZ, CachedQuadTreeNode node, int curNodeX, int curNodeZ, int curNodeScale, QuadTreeVisitor visitor) {
		if(node == null)
			return;
		
		assert node.getSize() == 1 << curNodeScale;
		
		QuadTreeVisitResult visitResult = visitor.visit(curNodeX, curNodeZ, curNodeScale, node);
		assert visitResult != null : "Visitor "+visitor+" returned null for node at "+curNodeX+"/"+curNodeZ+" (scale "+curNodeScale+")";
		if(visitResult == QuadTreeVisitResult.VISIT_CHILDREN) {
			assert curNodeScale > 0 : "Visitor "+visitor+" returned VISIT_CHILDREN for a leaf node.";
			
			// TODO encapsulate these getInt calls better
			// TODO: is there a better way to order these calls with less duplication?
			int childSize = 1 << (curNodeScale - 1);
			boolean renderPXFirst = refX < curNodeX+childSize;
			boolean renderPZFirst = refZ < curNodeZ+childSize;
			if(renderPZFirst) {
				if(renderPXFirst)
					visitQuadTreeBackToFront(refX, refZ, node.getChild(3), curNodeX+childSize, curNodeZ+childSize, curNodeScale-1, visitor);
				visitQuadTreeBackToFront(refX, refZ, node.getChild(2), curNodeX, curNodeZ+childSize, curNodeScale-1, visitor);
				if(!renderPXFirst)
					visitQuadTreeBackToFront(refX, refZ, node.getChild(3), curNodeX+childSize, curNodeZ+childSize, curNodeScale-1, visitor);
			}
			if(renderPXFirst)
				visitQuadTreeBackToFront(refX, refZ, node.getChild(1), curNodeX+childSize, curNodeZ, curNodeScale-1, visitor);
			visitQuadTreeBackToFront(refX, refZ, node.getChild(0), curNodeX, curNodeZ, curNodeScale-1, visitor);
			if(!renderPXFirst)
				visitQuadTreeBackToFront(refX, refZ, node.getChild(1), curNodeX+childSize, curNodeZ, curNodeScale-1, visitor);
			if(!renderPZFirst) {
				if(renderPXFirst)
					visitQuadTreeBackToFront(refX, refZ, node.getChild(3), curNodeX+childSize, curNodeZ+childSize, curNodeScale-1, visitor);
				visitQuadTreeBackToFront(refX, refZ, node.getChild(2), curNodeX, curNodeZ+childSize, curNodeScale-1, visitor);
				if(!renderPXFirst)
					visitQuadTreeBackToFront(refX, refZ, node.getChild(3), curNodeX+childSize, curNodeZ+childSize, curNodeScale-1, visitor);
			}
		}
	}
	


	/**
	 * Finds a node with the given coordinates and scale. If none exist, one will be created.
	 */
	public CachedQuadTreeNode getOrCreateNode(int x, int y, int scale) throws IOException {
		assert scale >= 0 && scale <= 30;
		int scaleMask = (1 << scale) - 1;
		assert (x & scaleMask) == 0;
		assert (y & scaleMask) == 0;
		
		int rootQuadrant = 0;
		if(x >= 0) {
			rootQuadrant |= 1;
		}
		if(y >= 0) {
			rootQuadrant |= 2;
		}
		
		int requiredRQScale = 32 - Integer.numberOfLeadingZeros(Math.max((x < 0 ? ~x - scaleMask : x), (y < 0 ? ~y - scaleMask : y))+scaleMask);
		ensureRootQuadrantScale(rootQuadrant, requiredRQScale);
		
		CachedQuadTreeNode curNode = getRootNode(rootQuadrant);
		int curNodeScale = quadTreeFile.base.getRootQuadrantScale(rootQuadrant);
		assert curNode != null;
		
		while(curNodeScale > scale) {
			int curNodeSize = 1 << curNodeScale;
			
			int childIndex = 0;
			if((x & (curNodeSize >> 1)) != 0)
				childIndex |= 1;
			if((y & (curNodeSize >> 1)) != 0)
				childIndex |= 2;
			
			curNode = curNode.getOrCreateChild(childIndex);
			assert curNode != null;
			curNodeScale--;
		}
		
		assert curNodeScale == scale;
		
		return curNode;
	}
	
	public void ensureRootQuadrantScale(int quadrant, int requiredScale) throws IOException {
		assert quadTreeFile.getRootQuadrantScale(quadrant) != 255;
		while(quadTreeFile.getRootQuadrantScale(quadrant) < requiredScale)
			expandRootQuadrant(quadrant);
	}

	private void expandRootQuadrant(int quadrant) throws IOException {
		CachedQuadTreeNode oldNode = getRootNode(quadrant);
		int newRootID = quadTreeFile.base.expandRootQuadrant(quadrant);
		CachedQuadTreeNode newNode = new CachedQuadTreeNode(newRootID, this, oldNode.getSize()<<1, null);
		
		// This will create a new CachedQuadTreeNode object for the old node. In a moment we'll
		// overwrite that with the same object we were using before (i.e. with oldNode).
		newNode.loadChildren();
		
		switch(quadrant) {
		case 0:
			rootNN = newNode;
			assert newNode.childPP.getNodeID() == oldNode.getNodeID();
			newNode.childPP = oldNode;
			break;
		case 1:
			rootPN = newNode;
			assert newNode.childNP.getNodeID() == oldNode.getNodeID();
			newNode.childNP = oldNode;
			break;
		case 2:
			rootNP = newNode;
			assert newNode.childPN.getNodeID() == oldNode.getNodeID();
			newNode.childPN = oldNode;
			break;
		case 3:
			rootPP = newNode;
			assert newNode.childNN.getNodeID() == oldNode.getNodeID();
			newNode.childNN = oldNode;
			break;
		default:
			throw new AssertionError("quadrant = "+quadrant);
		}
		oldNode.parent = newNode;
	}

	private CachedQuadTreeNode getRootNode(int rootQuadrant) {
		switch(rootQuadrant) {
		case 0: return rootNN;
		case 1: return rootPN;
		case 2: return rootNP;
		case 3: return rootPP;
		default: throw new RuntimeException("rootQuadrant "+rootQuadrant);
		}
	}

	public CachedQuadTreeNode getNode(int x, int z) {
		try {
			return getOrCreateNode(x, z, 0);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public void visitLoadedQuadTreeNodes(QuadTreeVisitor quadTreeVisitor) {
		visitLoadedQuadTreeNodes(quadTreeVisitor, rootNN);
		visitLoadedQuadTreeNodes(quadTreeVisitor, rootPN);
		visitLoadedQuadTreeNodes(quadTreeVisitor, rootNP);
		visitLoadedQuadTreeNodes(quadTreeVisitor, rootPP);
	}

	private void visitLoadedQuadTreeNodes(QuadTreeVisitor v, CachedQuadTreeNode n) {
		if(n == null)
			return;
		v.visit(0, 0, 0, n);
		visitLoadedQuadTreeNodes(v, n.childNN);
		visitLoadedQuadTreeNodes(v, n.childNP);
		visitLoadedQuadTreeNodes(v, n.childPN);
		visitLoadedQuadTreeNodes(v, n.childPP);
	}

	public void close() {
		// Data file must be closed before quadtree file, as data file operations can access the quadtree file.
		// So we need to wait in this thread until it's done.
		final Semaphore waitForDone = new Semaphore(1);
		try {
			waitForDone.acquire();
		} catch(InterruptedException e) {
			throw new AssertionError("Shouldn't happen", e);
		}
		InfiViewMod.ioThread.enqueueTrivial(new Runnable() {
			@Override
			public void run() {
				imageDataFile2.close();
				waitForDone.release();
			}
		});
		try {
			waitForDone.acquire();
		} catch(InterruptedException e) {
			throw new RuntimeException("Unexpected thread interrupt while waiting for data file to close.", e);
		}
		quadTreeFile.close();
	}
	
	
}
