package mods.immibis.infiview.storage;

import java.io.IOException;
import java.nio.ByteOrder;
import java.util.Arrays;

/**
 * Represents an open file storing an unbounded quadtree structure.
 * This class does not specify the payload of each node - rather, it's
 * wrapped in a {@link UnboundedQuadTreeFileL2} which interprets the payloads.
 * 
 * <pre>
 * File format:
 * The first 4096-byte block is reserved for the header.
 * Header byte 0: format version number.
 * Header byte 1-4: outermost quadrant scale, 0xFF if quadrant doesn't exist.
 * Header byte 5: Saved endianness. 1=big, 2=little.
 * Header byte 6-7: unused (0).
 * Header byte 8-11: nodeSizeBytes.
 * Header byte 12-15: index of next unused quadtree node.
 * 16-19: unused (0).
 * 20-23: index of -X -Y root, or -1 if none.
 * 24-27: index of +X -Y root, or -1 if none.
 * 28-31: index of -X +Y root, or -1 if none.
 * 32-35: index of +X +Y root, or -1 if none.
 * 
 * Quadtree nodes consist of:
 * Offset  0- 3: index of -X -Y child, or -1 if none.
 * Offset  4- 7: index of +X -Y child, or -1 if none.
 * Offset  8-11: index of -X +Y child, or -1 if none.
 * Offset 12-15: index of +X +Y child, or -1 if none.
 * Followed by nodeDataBytes additional bytes for application data.
 * On allocation, the node is filled with 0xFF bytes.
 * </pre>
 * 
 * TODO make non-public
 */
public final class UnboundedQuadTreeFileL1 {
	private int nodeSizeBytes;
	final ExpandableMemoryMappedFile file;
	
	public static final int NODE_FIXED_BYTES = 16; // number of node bytes not for application data
	
	public UnboundedQuadTreeFileL1(ExpandableMemoryMappedFile file, int nodeDataBytes) throws IOException {
		this.nodeSizeBytes = nodeDataBytes + NODE_FIXED_BYTES;
		if(nodeDataBytes < 0)
			throw new IllegalArgumentException("nodeDataBytes: "+nodeDataBytes);
		
		this.file = file;
		
		readHeader();
	}
	
	private int[] rootQuadrantScale = new int[4];
	private int[] rootQuadrantIndex = new int[4];
	private int nextUnusedNode;
	
	
	private void writeHeader() throws IOException {
		file.expandFileTo(4096);
		
		file.mapping.put(0, (byte)0);
		file.mapping.put(1, (byte)rootQuadrantScale[0]);
		file.mapping.put(2, (byte)rootQuadrantScale[1]);
		file.mapping.put(3, (byte)rootQuadrantScale[2]);
		file.mapping.put(4, (byte)rootQuadrantScale[3]);
		file.mapping.put(5, (byte)(ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN ? 1 : 2));
		file.mapping.putInt(8, nodeSizeBytes);
		file.mapping.putInt(12, nextUnusedNode);
		file.mapping.putInt(16, 0);
		file.mapping.putInt(20, rootQuadrantIndex[0]);
		file.mapping.putInt(24, rootQuadrantIndex[1]);
		file.mapping.putInt(28, rootQuadrantIndex[2]);
		file.mapping.putInt(32, rootQuadrantIndex[3]);
	}
	private void readHeader() throws IOException {
		if(file.expandFileTo(4096)) {
			Arrays.fill(rootQuadrantScale, -1);
			Arrays.fill(rootQuadrantIndex, -1);
			nextUnusedNode = 0;
			writeHeader();
			file.mapping.force();
			return;
		}
		
		int version = file.mapping.get(0);
		if(version != 0)
			throw new IOException("Unsupported file format version: "+version);
		
		int endianness = file.mapping.get(5);
		int ourEndianness = (ByteOrder.nativeOrder() == ByteOrder.BIG_ENDIAN ? 1 : 2);
		if(endianness != ourEndianness)
			throw new IOException("File was created on a system with endianness "+endianness+", but this system uses "+ourEndianness);
		
		int savedNodeSizeBytes = file.mapping.getInt(8);
		if(savedNodeSizeBytes != nodeSizeBytes)
			throw new IOException("Object constructed with "+nodeSizeBytes+" node bytes, but file uses "+savedNodeSizeBytes);
		
		nextUnusedNode = file.mapping.getInt(12);
		for(int k = 0; k < 4; k++) {
			rootQuadrantScale[k] = file.mapping.get(1 + k) & 255;
			rootQuadrantIndex[k] = file.mapping.getInt(20 + k*4);
		}
	}
	
	// TODO make private
	int getNodeDataPosition(int index) {
		return getNodePosition(index) + NODE_FIXED_BYTES;
	}
	
	// TODO make private
	public int getNodePosition(int index) {
		return 4096 + index*nodeSizeBytes;
	}

	private int allocateNode() throws IOException {
		if(getNodePosition(nextUnusedNode + 1) < 0) // check for overflow
			throw new AssertionError("Reached file size limit!");
		
		int nodeNum = nextUnusedNode++;
		file.mapping.putInt(12, nextUnusedNode);
		int pos = getNodePosition(nodeNum);
		file.expandFileTo(pos + nodeSizeBytes);
		
		file.mapping.position(pos);
		for(int k = 0; k < nodeSizeBytes; k ++)
			file.mapping.put((byte)0xFF);
		
		return nodeNum;
	}

	int expandRootQuadrant(int quadrant) throws IOException {
		// Creates a new node, with the old node as a child,
		// and sets the new node as the root quadrant.
		
		int oldIndex = rootQuadrantIndex[quadrant];
		int newIndex = allocateNode();
		
		int pos = getNodePosition(newIndex);
		
		file.mapping.putInt(pos + (3-quadrant)*4, oldIndex);
		rootQuadrantIndex[quadrant] = newIndex;
		if(rootQuadrantScale[quadrant] == 255)
			rootQuadrantScale[quadrant] = 0;
		else
			rootQuadrantScale[quadrant]++;
		file.mapping.putInt(20 + quadrant*4, newIndex);
		file.mapping.put(1 + quadrant, (byte)rootQuadrantScale[quadrant]);
		
		return newIndex;
	}

	public int getRootQuadrantScale(int i) {
		return rootQuadrantScale[i];
	}
	
	public int getRootQuadrantNodeID(int i) {
		return rootQuadrantIndex[i];
	}
	
	public void ensureRootNodeCreated(int quadrant) throws IOException {
		if(rootQuadrantScale[quadrant] == 255 || rootQuadrantScale[quadrant] == -1)
			expandRootQuadrant(quadrant);
	}
	
	public int allocateNewChild(int nodeID, int childNumber) throws IOException {
		assert childNumber >= 0 && childNumber <= 3;
		int childPtrPos = getNodePosition(nodeID) + childNumber*4;
		int newIndex = allocateNode();
		file.mapping.putInt(childPtrPos, newIndex);
		return newIndex;
	}
	
	public void close() {
		file.close();
	}
}
