package mods.immibis.redlogic.gates;

import net.minecraft.block.Block;
import net.minecraft.client.renderer.texture.IconRegister;
import net.minecraft.util.Icon;


public abstract class GateRendering {
	// Wire colours
	public static final int OFF = 0x400000;
	public static final int ON = 0xFF0000;
	public static final int DISABLED = 0xC0C0C0;
	
	public Icon torchTexOn = Block.torchRedstoneActive.getBlockTextureFromSide(0);
	public Icon torchTexOff = Block.torchRedstoneIdle.getBlockTextureFromSide(0);
	
	public String[] segmentTex = new String[] {"base"};
	public int[] segmentCol = new int[] {0xFF0000};
	
	// Torch positions are relative to the texture, in pixels
	public float[] torchX = new float[] {};
	public float[] torchY = new float[] {};
	public boolean[] torchState = new boolean[] {};
	
	// Ditto for pointer positions
	public float[] pointerX = new float[] {};
	public float[] pointerY = new float[] {};
	
	public void set(int renderState) {}
	public void setItemRender() {}
	
	public Icon[] segmentIcons = null;
	
	public void loadTextures(IconRegister register) {
		segmentIcons = new Icon[segmentTex.length];
		for(int k = 0; k < segmentTex.length; k++)
			segmentIcons[k] = register.registerIcon(ICON_PREFIX + segmentTex[k]);
	}
	
	public static final String ICON_PREFIX = "immibis/redlogic:";
	
	/**
	 * Marker interface for renderings that don't store any state.
	 * Similar to GateLogic.Stateless.
	 * Currently not used.
	 */
	public static interface Stateless {}
	
	public static class Default extends GateRendering implements Stateless {
		
	}
	
	public static class AND extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"and-base", "and-ovl-out", "and-ovl-back", "and-ovl-right", "and-ovl-left"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {8.5f, 4.5f, 8.5f, 12.5f};
			torchY = new float[] {2.5f, 6.5f, 6.5f, 6.5f};
			torchState = new boolean[] {false, true, true, true};
		}
		@Override
		public void set(int renderState) {
			boolean out_on = (renderState & 1) != 0;
			boolean back_on = (renderState & 2) != 0;
			boolean left_on = (renderState & 4) != 0;
			boolean right_on = (renderState & 8) != 0;
			boolean left_disabled = (renderState & 16) != 0;
			boolean back_disabled = (renderState & 32) != 0;
			boolean right_disabled = (renderState & 64) != 0;
			segmentCol[1] = out_on ? OFF : ON;
			segmentCol[2] = back_disabled ? DISABLED : back_on ? ON : OFF;
			segmentCol[3] = right_disabled ? DISABLED : right_on ? ON : OFF;
			segmentCol[4] = left_disabled ? DISABLED : left_on ? ON : OFF;
			torchState[0] = out_on;
			torchState[1] = !left_on && !left_disabled;
			torchState[2] = !back_on && !back_disabled;
			torchState[3] = !right_on && !right_disabled;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = ON;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = OFF;
			torchState[0] = false;
			torchState[1] = true;
			torchState[2] = true;
			torchState[3] = true;
		}
	}
	
	public static class OR extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"or-base", "or-ovl-out", "or-ovl-back", "or-ovl-right", "or-ovl-left", "or-ovl-middle"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0, 0xFF0000};
			torchX = new float[] {7.5f, 7.5f};
			torchY = new float[] {5.5f, 9.5f};
			torchState = new boolean[] {false, true};
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 64) != 0 ? DISABLED : (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 128) != 0 ? DISABLED : (renderState & 8) != 0 ? ON : OFF;
			segmentCol[4] = (renderState & 32) != 0 ? DISABLED : (renderState & 4) != 0 ? ON : OFF;
			segmentCol[5] = (renderState & 1) == 0 ? ON : OFF;
			torchState[0] = (renderState & 16) != 0;
			torchState[1] = !torchState[0];
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = OFF;
			segmentCol[5] = ON;
			torchState[0] = false;
			torchState[1] = true;
		}
	}
	
	public static class NOT extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"not-base", "not-ovl-out", "not-ovl-back", "not-ovl-right", "not-ovl-left"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {7.5f};
			torchY = new float[] {7.5f};
			torchState = new boolean[] {false};
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 64) != 0 ? DISABLED : (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 128) != 0 ? DISABLED : (renderState & 8) != 0 ? ON : OFF;
			segmentCol[4] = (renderState & 32) != 0 ? DISABLED : (renderState & 4) != 0 ? ON : OFF;
			torchState[0] = (renderState & 16) != 0;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = segmentCol[3] = segmentCol[4] = ON;
			segmentCol[2] = OFF;
			torchState[0] = true;
		}
	}
	
	public static class RSLatch extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"rs-base", "rs-left", "rs-right"};
			segmentCol = new int[] {0xFFFFFF, 0, 0};
			torchX = new float[] {6.5f, 9.5f};
			torchY = new float[] {3.5f, 12.5f};
			torchState = new boolean[] {false, false};
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = ON;
			segmentCol[2] = OFF;
			torchState[0] = true;
			torchState[1] = false;
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			torchState[0] = (renderState & 4) != 0;
			torchState[1] = (renderState & 8) != 0;
		}
	}
	
	public static class ToggleLatch extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"toggle-base", "toggle-left", "toggle-right"};
			segmentCol = new int[] {0xFFFFFF, 0, 0};
			torchX = new float[] {4.5f, 4.5f};
			torchY = new float[] {3.5f, 12.5f};
			torchState = new boolean[] {true, false};
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			torchState[0] = true;
			torchState[1] = false;
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			torchState[0] = (renderState & 4) != 0;
			torchState[1] = (renderState & 8) != 0;
		}
	}
	
	public static class NOR extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"nor-base", "nor-out", "nor-back", "nor-right", "nor-left"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {7.5f};
			torchY = new float[] {7.5f};
			torchState = new boolean[] {false};
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 64) != 0 ? DISABLED : (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 128) != 0 ? DISABLED : (renderState & 8) != 0 ? ON : OFF;
			segmentCol[4] = (renderState & 32) != 0 ? DISABLED : (renderState & 4) != 0 ? ON : OFF;
			torchState[0] = (renderState & 16) != 0;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = segmentCol[3] = segmentCol[4] = ON;
			segmentCol[2] = OFF;
			torchState[0] = true;
		}
	}
	
	public static class NAND extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"nand-base", "nand-out", "nand-back", "nand-right", "nand-left"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {4.5f, 8.5f, 12.5f};
			torchY = new float[] {6.5f, 6.5f, 6.5f};
			torchState = new boolean[] {true, true, true};
		}
		@Override
		public void set(int renderState) {
			boolean left_disabled = (renderState & 16) != 0;
			boolean back_disabled = (renderState & 32) != 0;
			boolean right_disabled = (renderState & 64) != 0;
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = back_disabled ? DISABLED : (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = right_disabled ? DISABLED : (renderState & 8) != 0 ? ON : OFF;
			segmentCol[4] = left_disabled ? DISABLED : (renderState & 4) != 0 ? ON : OFF;
			torchState[0] = (renderState & 4) == 0 && !left_disabled;
			torchState[1] = (renderState & 2) == 0 && !back_disabled;
			torchState[2] = (renderState & 8) == 0 && !right_disabled;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = ON;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = OFF;
			torchState[0] = true;
			torchState[1] = true;
			torchState[2] = true;
		}
	}
	
	public static class XOR extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"xor-base", "xor-left", "xor-right", "xor-middle", "xor-out"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {4.5f, 11.5f,  8.5f};
			torchY = new float[] {9.5f,  9.5f, 13.5f};
			torchState = new boolean[] {false, false, false};
		}
		@Override
		public void set(int renderState) {
			boolean left = (renderState & 1) != 0;
			boolean right = (renderState & 2) != 0;
			boolean out = (renderState & 4) != 0;
			segmentCol[1] = left ? ON : OFF;
			segmentCol[2] = right ? ON : OFF;
			segmentCol[3] = !left && !right ? ON : OFF;
			segmentCol[4] = out ? ON : OFF;
			torchState[0] = !left && right;
			torchState[1] = left && !right;
			torchState[2] = !left && !right;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = ON;
			segmentCol[4] = OFF;
			torchState[0] = false;
			torchState[1] = false;
			torchState[2] = true;
		}
	}
	
	public static class XNOR extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"xnor-base", "xnor-left", "xnor-right", "xnor-middle", "xnor-left-out", "xnor-right-out"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0, 0};
			torchX = new float[] {8.0f, 4.5f, 11.5f,  8.5f};
			torchY = new float[] {3.0f, 9.5f,  9.5f, 13.5f};
			torchState = new boolean[] {false, false, false, false};
		}
		@Override
		public void set(int renderState) {
			boolean left = (renderState & 1) != 0;
			boolean right = (renderState & 2) != 0;
			boolean out = (renderState & 4) != 0;
			segmentCol[1] = left ? ON : OFF;
			segmentCol[2] = right ? ON : OFF;
			segmentCol[3] = !left && !right ? ON : OFF;
			segmentCol[4] = !left && right ? ON : OFF;
			segmentCol[5] = left && !right ? ON : OFF;
			torchState[0] = out;
			torchState[1] = !left && right;
			torchState[2] = left && !right;
			torchState[3] = !left && !right;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = ON;
			segmentCol[4] = OFF;
			segmentCol[5] = OFF;
			torchState[0] = true;
			torchState[1] = false;
			torchState[2] = false;
			torchState[3] = true;
		}
	}
	
	public static class Buffer extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"buffer-base", "buffer-left", "buffer-right", "buffer-back", "buffer-out"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {7.5f, 7.5f};
			torchY = new float[] {2.5f, 9.5f};
			torchState = new boolean[] {false, false};
		}
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 4) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 8) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 2) != 0 ? ON : OFF;
			segmentCol[4] = (renderState & 2) == 0 ? ON : OFF;
			torchState[0] = (renderState & 1) != 0;
			torchState[1] = (renderState & 2) == 0;
		}
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = ON;
			torchState[0] = false;
			torchState[1] = true;
		}
	}
	
	public static class Multiplexer extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"multiplexer-base", "multiplexer-2", "multiplexer-3", "multiplexer-right", "multiplexer-5", "multiplexer-left-out", "multiplexer-right-out"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0, 0, 0};
			torchX = new float[] {8f, 4.5f, 11.5f, 4.5f};
			torchY = new float[] {2f, 7.5f, 7.5f, 12.5f};
			torchState = new boolean[] {false, false, false, false};
		}
		
		@Override
		public void set(int renderState) {
			boolean back = (renderState & 1) != 0;
			boolean left = (renderState & 2) != 0;
			boolean right = (renderState & 4) != 0;
			boolean out = (renderState & 8) != 0;
			segmentCol[1] = back ? ON : OFF;
			segmentCol[2] = left ? ON : OFF;
			segmentCol[3] = right ? ON : OFF;
			segmentCol[4] = !back ? ON : OFF;
			segmentCol[5] = !left && back ? ON : OFF;
			segmentCol[6] = !right && !back ? ON : OFF;
			torchState[0] = out;
			torchState[1] = !left && back;
			torchState[2] = !right && !back;
			torchState[3] = !back;
		}
		
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = ON;
			segmentCol[5] = OFF;
			segmentCol[6] = ON;
			torchState[0] = false;
			torchState[1] = false;
			torchState[2] = true;
			torchState[3] = true;
		}
	}
	
	public static class Repeater extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"repeater-base", "repeater-strip"};
			segmentCol = new int[] {0xFFFFFF, 0};
			torchX = new float[] {8f, 8f};
			torchY = new float[] {3f, 6f};
			torchState = new boolean[] {false, false};
		}
		
		@Override
		public void set(int renderState) {
			boolean out = (renderState & 32768) != 0;
			
			torchY[1] = (renderState & 7) + 6;
			
			torchState[0] = torchState[1] = out;
			
			segmentCol[1] = out ? ON : OFF;
		}
		
		@Override
		public void setItemRender() {
			torchY[1] = 6f;
			
			torchState[0] = torchState[1] = false;
			
			segmentCol[1] = OFF;
		}
	}
	
	public static class Timer extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"timer-base", "timer-left", "timer-back", "timer-right"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0};
			torchX = new float[] {8f};
			torchY = new float[] {2f};
			torchState = new boolean[] {false};
			pointerX = new float[] {8f};
			pointerY = new float[] {8f};
		}
		
		@Override
		public void set(int renderState) {
			torchState[0] = (renderState & 1) != 0;
			segmentCol[1] = (renderState & 2) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 4) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 8) != 0 ? ON : OFF;
		}
		
		@Override
		public void setItemRender() {
			torchState[0] = false;
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
		}
	}
	
	public static class Counter extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"counter-base", "counter-front", "counter-back"};
			segmentCol = new int[] {0xFFFFFF, 0, 0};
			torchX = new float[] {3f, 13f};
			torchY = new float[] {8f, 8f};
			torchState = new boolean[] {false, false};
			pointerX = new float[] {8f};
			pointerY = new float[] {11f};
		}
		
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			torchState[0] = (renderState & 4) != 0;
			torchState[1] = (renderState & 8) != 0;
		}
		
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			torchState[0] = true;
			torchState[1] = false;
		}
	}
	
	public static class Sequencer extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"sequencer-base"};
			segmentCol = new int[] {0xFFFFFF};
			torchX = new float[] {8f, 2f, 14f, 8f};
			torchY = new float[] {2f, 8f, 8f, 14f};
			torchState = new boolean[] {false, false, false, false};
			pointerX = new float[] {8f};
			pointerY = new float[] {8f};
		}
		
		@Override
		public void set(int renderState) {
			torchState[0] = renderState == 0;
			torchState[1] = renderState == 3;
			torchState[2] = renderState == 1;
			torchState[3] = renderState == 2;
		}
		
		@Override
		public void setItemRender() {
			torchState[0] = true;
			torchState[1] = false;
			torchState[2] = false;
			torchState[3] = false;
		}
	}
	
	public static class PulseFormer extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"former-base", "former-in", "former-3", "former-4", "former-5"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0};
			torchX = new float[] {8f, 4.5f, 11.5f};
			torchY = new float[] {2f, 8f, 8f};
			torchState = new boolean[] {false, false, false};
		}
		
		@Override
		public void set(int renderState) {
			boolean in = (renderState & 1) != 0;
			boolean out = (renderState & 2) != 0;
			boolean changing = (renderState & 4) != 0;
			torchState[0] = out;
			torchState[1] = !in;
			torchState[2] = in && (!out || changing);
			segmentCol[1] = in ? ON : OFF;
			segmentCol[2] = in ? OFF : ON;
			segmentCol[3] = in ? OFF : ON;
			segmentCol[4] = /*in && !out*/ torchState[2] ? ON : OFF;
		}
		
		@Override
		public void setItemRender() {
			torchState[0] = false;
			torchState[1] = true;
			torchState[2] = false;
			segmentCol[1] = OFF;
			segmentCol[2] = ON;
			segmentCol[3] = ON;
			segmentCol[4] = OFF;
		}
	}
	
	public static class Randomizer extends GateRendering {
		{
			torchX = new float[] {8f, 3f, 13f};
			torchY = new float[] {3f, 8f, 8f};
			torchState = new boolean[] {false, false, false};
			segmentTex = new String[] {"randomizer-base", "randomizer-in"};
			segmentCol = new int[] {0xFFFFFF, 0};
		}
		
		@Override
		public void loadTextures(IconRegister register) {
			super.loadTextures(register);
			
			torchTexOn = register.registerIcon(ICON_PREFIX+"randomizer-torch-on");
			torchTexOff = register.registerIcon(ICON_PREFIX+"randomizer-torch-off");
		}
		
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			torchState[0] = (renderState & 8) != 0;
			torchState[1] = (renderState & 2) != 0;
			torchState[2] = (renderState & 4) != 0;
		}
		
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			torchState[0] = false;
			torchState[1] = false;
			torchState[2] = false;
		}
	}
	
	public static class StateCell extends GateRendering implements Stateless {
		{
			segmentTex = new String[] {"statecell-base", "statecell-2", "statecell-3", "statecell-4", "statecell-5", "statecell-6", "statecell-7"};
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0, 0, 0};
			torchX = new float[] {13f};
			torchY = new float[] {7f};
			torchState = new boolean[] {false};
			pointerX = new float[] {8f};
			pointerY = new float[] {12f};
		}
		
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 4) != 0 ? ON : OFF;
			segmentCol[4] = (renderState & 16) == 0 ? ON : OFF;
			segmentCol[5] = (renderState & 16) != 0 ? ON : OFF;
			segmentCol[6] = (renderState & 8) != 0 ? ON : OFF;
			torchState[0] = (renderState & 8) != 0;
		}
		
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = ON;
			segmentCol[5] = OFF;
			segmentCol[6] = OFF;
			torchState[0] = false;
		}
	}
	
	public static class Synchronizer extends GateRendering implements Stateless {
		{
			segmentCol = new int[] {0xFFFFFF, 0, 0, 0, 0, 0, 0};
			segmentTex = new String[] {"sync-base", "sync-left", "sync-right", "sync-back", "sync-middle", "sync-left-middle", "sync-right-middle"};
			torchX = new float[] {8f};
			torchY = new float[] {3f};
			torchState = new boolean[] {false};
		}
		
		@Override
		public void set(int renderState) {
			segmentCol[1] = (renderState & 1) != 0 ? ON : OFF;
			segmentCol[2] = (renderState & 2) != 0 ? ON : OFF;
			segmentCol[3] = (renderState & 4) != 0 ? ON : OFF;
			torchState[0] = (renderState & 8) != 0;
			segmentCol[4] = (renderState & 48) != 0 ? ON : OFF;
			segmentCol[5] = (renderState & 16) != 0 ? OFF : ON;
			segmentCol[6] = (renderState & 32) != 0 ? OFF : ON;
		}
		
		@Override
		public void setItemRender() {
			segmentCol[1] = OFF;
			segmentCol[2] = OFF;
			segmentCol[3] = OFF;
			segmentCol[4] = OFF;
			segmentCol[5] = ON;
			segmentCol[6] = ON;
			torchState[0] = false;
		}
	}
}
