package mods.immibis.redlogic.wires;

import java.util.Arrays;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.ChatComponentText;
import net.minecraftforge.common.util.ForgeDirection;
import mods.immibis.redlogic.CommandDebug;
import mods.immibis.redlogic.api.wiring.IBundledEmitter;
import mods.immibis.redlogic.api.wiring.IBundledUpdatable;
import mods.immibis.redlogic.api.wiring.IBundledWire;
import mods.immibis.redlogic.api.wiring.IRedstoneUpdatable;

public class BundledTile extends WireTile implements IRedstoneUpdatable, IBundledEmitter, IBundledUpdatable, IBundledWire {
	
	private byte[] strength = new byte[16];
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		
		tag.setByteArray("strength", strength);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		
		strength = tag.getByteArray("strength");
	}
	
	@Override
	protected boolean canConnectToWire(WireTile wire) {
		return super.canConnectToWire(wire) || wire instanceof InsulatedRedAlloyTile;
	}

	@Override
	public boolean canUpdate() {
		return false;
	}
	
	private byte[] oldStrength = new byte[16];
	
	private short isUpdating = 0;
	private short recursiveUpdatePending = -1;
	
	private void updateStrengthFromBlock(int x, int y, int z, int side, int dir) {
		TileEntity te = worldObj.getTileEntity(x, y, z);
		
		if(te instanceof InsulatedRedAlloyTile) {
			InsulatedRedAlloyTile o = (InsulatedRedAlloyTile)te;
			int o_strength = o.getRedstoneSignalStrength() - 1;
			int colour = o.getInsulatedWireColour();
			if((strength[colour] & 0xFF) < o_strength)
				strength[colour] = (byte)o_strength;
			
		} else if(te instanceof IBundledEmitter) {
			byte[] o_strength = ((IBundledEmitter)te).getBundledCableStrength(side, dir);
			if(o_strength != null) {
				// null = all 0
				
				for(int k = 0; k < 16; k++) {
					int o_c_strength = (o_strength[k] & 0xFF) - 1;
					if((strength[k] & 0xFF) < o_c_strength)
						strength[k] = (byte)o_c_strength;
				}
			}
		}
	}
	
	private void updateStrengthFromSurroundingBlocks() {
		{
			byte[] temp = oldStrength;
			oldStrength = strength;
			strength = temp;
			Arrays.fill(strength, (byte)0);
		}
		
		for(int side = 0; side < 6; side++) {
			for(int dir = 0; dir < 6; dir++) {
				if(connectsInDirection(side, dir)) {
					ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[dir];
					int x = xCoord + fd.offsetX, y = yCoord + fd.offsetY, z = zCoord + fd.offsetZ;
					
					int oside = side, odir = dir^1;
					
					if(connectsInDirectionAroundCorner(side, dir)) {
						fd = ForgeDirection.VALID_DIRECTIONS[side];
						x += fd.offsetX;
						y += fd.offsetY;
						z += fd.offsetZ;
						
						oside = dir ^ 1; odir = side ^ 1;
					}
					
					updateStrengthFromBlock(x, y, z, oside, odir);
				}
			}
			
			if(connectsInDirectionByJacketedWire(side)) {
				ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[side];
				int x = xCoord + fd.offsetX, y = yCoord + fd.offsetY, z = zCoord + fd.offsetZ;
				
				updateStrengthFromBlock(x, y, z, -1, side ^ 1);
			}
		}
	}
	
	private void updateStrength(short updatingColours) {
		if(worldObj.isRemote)
			return;
		
		short outerUpdatingColours = isUpdating;
		
		// XXX THIS IS TERRIBLY NOT-UNDERSTANDABLE CODE
		// HINT: This is basically RedAlloyTile.updateStrength() but for up to 16 wires in parallel
		// The bitmask updatingColours selects which wires this is being called for.
		// The fields recursiveUpdatePending and isUpdating are replaced with bitmasks.
		
		recursiveUpdatePending |= (updatingColours & outerUpdatingColours);
		updatingColours &= ~outerUpdatingColours;
		
		if(updatingColours == 0)
			return;
		
		if(CommandDebug.WIRE_DEBUG_PARTICLES)
			debugEffect_bonemeal();
		
		isUpdating |= updatingColours;
		
		//System.out.println(xCoord+" "+yCoord+" "+zCoord+" "+updatingColours);
		
		//try {
			do {
				updatingColours |= (recursiveUpdatePending & ~outerUpdatingColours);
				recursiveUpdatePending &= outerUpdatingColours;
				
				//System.out.println(xCoord+" "+yCoord+" "+zCoord+" inner "+updatingColours);
				
				updateStrengthFromSurroundingBlocks();
				
				short decreased_mask = 0;
				short different_mask = 0;
				for(int k = 0; k < 16; k++) {
					if(strength[k] != oldStrength[k])
						different_mask |= 1 << k;
					if((strength[k] & 0xFF) < (oldStrength[k] & 0xFF)) {
						decreased_mask |= 1 << k;
						strength[k] = 0;
					}
				}
				
				decreased_mask &= updatingColours;
				different_mask &= updatingColours;
				
				if(decreased_mask != 0) {
					
					updateConnectedThings(decreased_mask);
					updateStrengthFromSurroundingBlocks();
					
					different_mask = decreased_mask;
					for(int k = 0; k < 16; k++)
						if(strength[k] != oldStrength[k])
							different_mask |= 1 << k;
				}
				
				if(different_mask != 0)
					updateConnectedThings(different_mask);
				
			} while((recursiveUpdatePending & ~outerUpdatingColours) != 0);
		//} catch(StackOverflowError e) {
		//	e.printStackTrace();
		//	System.err.println("Got stack overflow, aborting!");
		//}
		
		isUpdating &= ~updatingColours;
	}
	
	private void updateConnectedThings(short colourMask) {
		int notifiedSides = 0;
		
		if(CommandDebug.WIRE_DEBUG_PARTICLES)
			debugEffect_bonemeal();
		
		for(int side = 0; side < 6; side++) {
			for(int dir = 0; dir < 6; dir++) {
				if(connectsInDirection(side, dir)) {
					ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[dir];
					int x = xCoord + fd.offsetX, y = yCoord + fd.offsetY, z = zCoord + fd.offsetZ;
					
					if(connectsInDirectionAroundCorner(side, dir)) {
						fd = ForgeDirection.VALID_DIRECTIONS[side];
						x += fd.offsetX;
						y += fd.offsetY;
						z += fd.offsetZ;
					
					} else {
						if((notifiedSides & (1 << dir)) != 0)
							continue;
						notifiedSides |= 1 << dir;
					}
					
					TileEntity t = worldObj.getTileEntity(x, y, z);
					if(t instanceof BundledTile)
						((BundledTile)t).updateStrength(colourMask);
					else if(t instanceof IBundledUpdatable)
						((IBundledUpdatable)t).onBundledInputChanged();
				}
			}
			
			if(connectsInDirectionByJacketedWire(side)) {
				if((notifiedSides & (1 << side)) == 0) {
					notifiedSides |= 1 << side;
					
					ForgeDirection fd = ForgeDirection.VALID_DIRECTIONS[side];
					int x = xCoord + fd.offsetX, y = yCoord + fd.offsetY, z = zCoord + fd.offsetZ;
					
					TileEntity t = worldObj.getTileEntity(x, y, z);
					if(t instanceof IBundledUpdatable)
						((IBundledUpdatable)t).onBundledInputChanged();
				}
			}
		}
	}

	@Override
	public void onRedstoneInputChanged() {
		updateStrength((short)-1);
	}
	
	@Override
	public void onBundledInputChanged() {
		updateStrength((short)-1);
	}
	
	@Override
	void onNeighbourBlockChange() {
		super.onNeighbourBlockChange();
		updateStrength((short)-1);
	}
	
	@Override
	public byte[] getBundledCableStrength(int blockFace, int direction) {
		return connectsInDirection(blockFace, direction) ? strength : null;
	}
	
	@Override
	protected boolean debug(EntityPlayer ply) {
		if(!worldObj.isRemote)
		{
			int[] i = new int[16];
			for(int k = 0; k < 16; k++)
				i[k] = strength[k] & 0xFF;
			ply.addChatMessage(new ChatComponentText("Bundled cable strength: " + Arrays.toString(i)));
		}
		return true;
	}
}
