package mods.immibis.chunkloader;

import mods.immibis.core.BasicInventory;
import mods.immibis.core.api.net.IPacket;
import mods.immibis.core.api.porting.SidedProxy;
import mods.immibis.core.api.util.BaseContainer;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.Slot;
import net.minecraft.item.ItemStack;

public class ContainerChunkLoader extends BaseContainer<TileChunkLoader> {
	
	public EntityPlayer player;
	public TileChunkLoader tile;
	
	public final boolean useFuel;
	
	public ContainerChunkLoader(EntityPlayer player, final TileChunkLoader tile, boolean useFuel) {
		super(player, tile);
		this.player = player;
		this.tile = tile;
		this.useFuel = useFuel;
		
		if(useFuel) {
			for(int x = 0; x < 9; x++)
				addSlotToContainer(new Slot(player.inventory, x, 13 + 18*x, 178));
			for(int y = 0; y < 3; y++)
				for(int x = 0; x < 9; x++)
					addSlotToContainer(new Slot(player.inventory, x + y*9 + 9, 13 + 18*x, 120 + 18*y));
			
			addSlotToContainer(new Slot((IInventory)tile, 0, 192, 48));
		}
	}
	
	@Override
	public ItemStack transferStackInSlot(int slot) {
		if(slot < 36)
			BasicInventory.mergeStackIntoRange(player.inventory, (IInventory)tile, slot, 0, 1);
		else
			BasicInventory.mergeStackIntoRange((IInventory)tile, player.inventory, 0, 0, 36);
		return null;
	}
	
	public boolean canEdit() {
		return tile.owner != null && ((ownedByMe && !tile.isServerOwned) || amIOp);
	}
	
	@Override
	public void onButtonPressed(int i) {
		if(!canEdit())
			return;
		if(i == 3) {
			// decrease size
			if(tile.radius > 0) {
				tile.radius--;
				tile.loaderChanged(player.username);
			}
		} else if(i == 4) {
			// increase size
			if(tile.radius < DimensionalAnchors.MAX_RADIUS) {
				tile.radius++;
				tile.loaderChanged(player.username);
				tile.loadChunks();
			}
		} else if(i == 5 && amIOp) {
			tile.isServerOwned = !tile.isServerOwned;
			if(!tile.isServerOwned)
				tile.owner = player.username;
			tile.loaderChanged(player.username);
		
		} else if(i == 6) {
			tile.shape = Shape.VALUES[(tile.shape.ordinal() + 1) % Shape.VALUES.length];
			tile.loaderChanged(player.username);
		}
	}
	
	private String lastOwner = null;
	private int lastMaxQuota = -1, lastCurQuota = -1, lastRadius = -2;
	private boolean lastAmIOp, lastServerOwned;
	
	public boolean ownedByMe, amIOp;
	private int maxQuota, curQuota;
	
	public int getMaxQuota() {
		if(tile.worldObj.isRemote)
			return maxQuota;
		return tile.owner == null ? 0 : DimensionalAnchors.instance.getMaxQuota(tile.owner);
	}
	public int getCurQuota() {
		if(tile.worldObj.isRemote)
			return curQuota;
		return tile.owner == null ? 0 : DimensionalAnchors.instance.getCurQuota(tile.owner);
	}
	
	public void updateFields() {
		maxQuota = getMaxQuota();
		curQuota = getCurQuota();
		ownedByMe = (tile.owner != null && tile.owner.equals(player.username));
		amIOp = SidedProxy.instance.isOp(player.username);
		fuelEndTime = tile.calcFuelEndTime();
	}
	
	private int lastScaledFuelTime = -1, lastMaxFuelTime = -1;
	
	@Override
	public void updateProgressBar(int id, int val) {
		if(id == 0)
			tile.currentFuelTicks = val;
		else if(id == 1)
			tile.remainingFuelTicks = val;
	}
	
	private Shape lastShape;
	
	public long fuelEndTime;
	private long lastFuelEndTime;
	
	@Override
	public void detectAndSendChanges() {
		super.detectAndSendChanges();
		
		updateFields();
		
		if(lastMaxFuelTime != 14) {
			lastMaxFuelTime = 14;
			sendProgressBarUpdate(0, lastMaxFuelTime);
		}
		
		int scaledFuelTime = tile.getScaledFuelTime(lastMaxFuelTime);
		if(lastScaledFuelTime != scaledFuelTime) {
			lastScaledFuelTime = scaledFuelTime;
			sendProgressBarUpdate(1, scaledFuelTime);
		}
	
		if(lastOwner == tile.owner && lastMaxQuota == maxQuota && lastCurQuota == curQuota && lastRadius == tile.radius && lastAmIOp == amIOp && lastServerOwned == tile.isServerOwned && lastShape == tile.shape && fuelEndTime == lastFuelEndTime) {
			return;
		}
		
		PacketGUIUpdate packet = new PacketGUIUpdate(tile.owner == null ? "" : tile.owner, curQuota, maxQuota, tile.radius, amIOp, tile.isServerOwned, tile.shape, fuelEndTime);

		sendUpdatePacket(packet);
		
		lastOwner = tile.owner;
		lastCurQuota = curQuota;
		lastMaxQuota = maxQuota;
		lastRadius = tile.radius;
		lastAmIOp = amIOp;
		lastServerOwned = tile.isServerOwned;
		lastShape = tile.shape;
		lastFuelEndTime = fuelEndTime;
	}
	
	@Override
	public void onUpdatePacket(IPacket packet) {
		if(packet instanceof PacketGUIUpdate) {
			PacketGUIUpdate p = (PacketGUIUpdate)packet;
			tile.owner = p.owner.equals("") ? null : p.owner;
			tile.radius = p.radius;
			curQuota = p.curQuota;
			maxQuota = p.maxQuota;
			amIOp = p.amIOp;
			tile.isServerOwned = p.isServerOwned;
			ownedByMe = (tile.owner != null && tile.owner.equals(player.username));
			tile.shape = Shape.VALUES[p.shapeID];
			fuelEndTime = p.fuelEndTime;
		}
	}
	
}
