package mods.immibis.chunkloader;

import mods.immibis.core.TileCombined;
import mods.immibis.core.api.traits.IInventoryTraitUser;
import mods.immibis.core.api.traits.IInventoryTrait;
import mods.immibis.core.api.traits.TraitField;
import mods.immibis.core.api.traits.UsesTraits;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.world.ChunkCoordIntPair;
import net.minecraft.world.World;
import net.minecraft.world.chunk.EmptyChunk;
import net.minecraft.world.chunk.IChunkProvider;

@UsesTraits
public class TileChunkLoader extends TileCombined implements IInventoryTraitUser {
	// Trialing this traits system
	@TraitField public IInventoryTrait inv;
	
	public WorldInfo worldInfo;
	
	public String owner;
	public int radius = 0;
	public boolean isServerOwned = false;
	
	public Shape shape = Shape.SQUARE;
	
	private boolean wasPreviouslyInvalidated = false;
	
	public int currentFuelTicks = 1;	// number of ticks the last fuel item lasts for, never 0
	public int remainingFuelTicks = 0;	// number of ticks left from the last fuel item(s), can be >= currentFuelTicks
	
	public int getScaledFuelTime(int max) {
		if(remainingFuelTicks >= currentFuelTicks)
			return max;
		return remainingFuelTicks * max / currentFuelTicks;
	}
	
	private boolean isLoaderActive;
	public void setActive(boolean active) {
		if(active != isLoaderActive) {
			if(active) {
				limitRadius();
				worldInfo.addLoader(this);
			} else
				worldInfo.removeLoader(this);
			isLoaderActive = active;
		}
	}
	
	public boolean canLoaderBeActive() {
		return !DimensionalAnchors.requireFuel || isServerOwned || remainingFuelTicks >= getNumChunks();
	}
	
	private boolean useFuelItem() {
		ItemStack is = inv.getStackInSlot(0);
		
		if(is == null)
			return false;
		
		int value = Fuels.get(is.itemID, is.getItemDamage());
		if(value <= 0)
			return false;
		
		//System.out.println("Adding "+value+" ticks");
		
		currentFuelTicks = value;
		remainingFuelTicks += value;
		
		is.stackSize--;
		if(is.stackSize <= 0)
			is = null;
		
		inv.setInventorySlotContents(0, is);
		
		return true;
	}
	
	// For rendering
	float spinProgress;
	
	@Override
	public void updateEntity() {
		
		spinProgress += Math.PI / 20;
		
		if(worldObj.isRemote)
			return;
		
		if(isServerOwned)
			setActive(true);
		else if(DimensionalAnchors.requireFuel) {
			int usedFuel = getNumChunks();
			
			while(remainingFuelTicks < usedFuel && useFuelItem()) {
			}
			
			if(remainingFuelTicks < usedFuel) {
				remainingFuelTicks = 0;
				setActive(false);
			} else {
				remainingFuelTicks -= usedFuel;
				setActive(true);
				//System.out.println("Used "+usedFuel+" ticks, "+remainingFuelTicks+" remaining");
			}
		}
		
		if(DEBUG)
			System.out.println("loader ticking in world "+worldObj.provider.dimensionId);
	}
	
	private static final boolean DEBUG = World.class.getSimpleName().equals("World") && false;
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setString("owner", owner == null ? "" : owner);
		tag.setInteger("radius", radius);
		tag.setBoolean("serverOwned", isServerOwned);
		tag.setInteger("rfuel", remainingFuelTicks);
		tag.setInteger("mfuel", currentFuelTicks);
		tag.setInteger("shape", shape.ordinal());
		
		inv.writeToNBT(tag);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		owner = tag.getString("owner");
		if(owner != null && owner.equals(""))
			owner = null;
		radius = tag.getInteger("radius");
		isServerOwned = tag.getBoolean("serverOwned");
		remainingFuelTicks = tag.getInteger("rfuel");
		currentFuelTicks = tag.getInteger("mfuel");
		shape = Shape.VALUES[tag.getInteger("shape")];
		
		if(currentFuelTicks < 1) currentFuelTicks = 1;
		
		inv.readFromNBT(tag);
	}
	
	@Override
	public boolean onBlockActivated(EntityPlayer ply) {
		if(!ply.worldObj.isRemote)
			ply.openGui(DimensionalAnchors.instance, DimensionalAnchors.requireFuel ? DimensionalAnchors.GUI_CHUNKLOADER_FUELED : DimensionalAnchors.GUI_CHUNKLOADER, worldObj, xCoord, yCoord, zCoord);
		return true;
	}
	
	@Override
	public void validate() {
		try {
			super.validate();
			if(worldObj.isRemote)
				return;
			
			worldInfo = DimensionalAnchors.getWorld(worldObj);
			
			if(canLoaderBeActive()) {
				setActive(true);
				if(wasPreviouslyInvalidated)
					loadChunks();
			} else
				setActive(false);
			
			wasPreviouslyInvalidated = false;
			if(owner != null)
				Logging.onValidated(getLoaderInfo(), "<unknown>");
		} catch(Throwable t) {
			String message = "Dimensional Anchors: TileChunkLoader.validate threw an exception. " + 
				"Chunk loader coords: "+xCoord+","+yCoord+","+zCoord+" in dimension "+worldObj.provider.dimensionId;
			new Exception(message, t).printStackTrace();
		}
	}
	@Override
	public void invalidate() {
		super.invalidate();
		if(worldObj.isRemote)
			return;
		
		if(isLoaderActive) {
			worldInfo.delayRemoveLoader(this);
			isLoaderActive = false;
		}
		if(owner != null)
			Logging.onInvalidated(getLoaderInfo(), "<unknown>");
		wasPreviouslyInvalidated = true;
	}
	
	private void loaderChanged() {
		if(isLoaderActive) {
			setActive(false);
			setActive(true);
		}
	}
	
	public void loaderChanged(String byWho) {
		loaderChanged(byWho, false);
	}
	
	public void loaderChanged(String byWho, boolean justAdded) {
		loaderChanged();
		if(justAdded)
			Logging.onAdded(getLoaderInfo(), byWho);
		else
			Logging.onChanged(getLoaderInfo(), byWho);
	}
	
	void loadChunks() {
		IChunkProvider provider = worldObj.getChunkProvider();
		for(ChunkCoordIntPair ccip : getLoaderInfo().getLoadedChunks())
			if(provider.provideChunk(ccip.chunkXPos, ccip.chunkZPos) instanceof EmptyChunk)
				provider.loadChunk(ccip.chunkXPos, ccip.chunkZPos);
	}
	
	public void limitRadius() {
		if(owner == null) {
			radius = -1;
			return;
		}
		
		int max = DimensionalAnchors.instance.getMaxQuota(owner);
		
		if(max == DimensionalAnchors.UNLIMITED)
			return;
		
		int cur = DimensionalAnchors.instance.getCurQuota(owner);
		
		while(radius >= 0 && cur + getNumChunks() > max) {
			radius--;
			loaderChanged("<enforcing quota limit for "+owner+">");
		}
		
		if(cur + getNumChunks() > max)
			radius--;
	}
	
	public int getNumChunks() {
		if(radius < 0)
			return 0;
		return shape.getNumChunks(radius);
	}

	public WorldInfo.LoaderInfo getLoaderInfo() {
		WorldInfo.LoaderInfo li = new WorldInfo.LoaderInfo(new WorldInfo.XYZ(this), worldInfo, owner, radius, shape);
		li.isServerOwned = isServerOwned;
		return li;
	}

	public long calcFuelEndTime() {
		int usedFuelPerTick = getNumChunks();
		
		int availableFuel = remainingFuelTicks;
		
		ItemStack fuelStack = inv.getStackInSlot(0);
		if(fuelStack != null) {
			int fuelPerItem = Fuels.get(fuelStack.itemID, fuelStack.getItemDamage());
			if(fuelPerItem > 0)
				availableFuel += fuelPerItem * fuelStack.stackSize;
		}
		
		if(usedFuelPerTick == 0)
			return worldObj.getTotalWorldTime();
		
		return worldObj.getTotalWorldTime() + availableFuel / usedFuelPerTick;
	}
	
	
	

	private int[] as_slot0 = new int[] {0};
	private int[] as_none = new int[0];
	@Override
	public int[] getAccessibleSlots(int side) {
		return DimensionalAnchors.requireFuel ? as_slot0 : as_none;
	}

	@Override
	public boolean canInsert(int slot, int side, ItemStack stack) {
		return DimensionalAnchors.allowFuelPiping;
	}

	@Override
	public boolean canInsert(int slot, ItemStack stack) {
		return true;
	}

	@Override
	public boolean canExtract(int slot, int side, ItemStack stack) {
		return false;
	}
	
	@Override
	public int getInventorySize() {
		return 1;
	}
}
