package mods.immibis.chunkloader;

import java.io.File;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.world.ChunkCoordIntPair;
import net.minecraft.world.World;
import net.minecraft.world.WorldSavedData;
import net.minecraft.world.WorldServer;

public class WorldInfo extends WorldSavedData {
	
	public WeakReference<WorldServer> worldRef;
	public Object cliData;
	
	public static class XYZ {
		int x, y, z;
		public XYZ(int x, int y, int z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}
		@Override
		public int hashCode() {
			return x * 23434 + y * 2351321 + z;
		}
		@Override
		public boolean equals(Object o) {
			try {
				XYZ _o = (XYZ)o;
				return x == _o.x && y == _o.y && z == _o.z;
			} catch(ClassCastException e) {
				return false;
			}
		}
		public XYZ(TileEntity tile) {
			this(tile.xCoord, tile.yCoord, tile.zCoord);
		}
		
		@Override
		public String toString() {
			return "["+x+","+y+","+z+"]";
		}
	}
	
	public static class LoaderInfo {
		public LoaderInfo(XYZ pos, WorldInfo world, String player, int radius, Shape shape) {
			this.pos = pos;
			this.removeTime = -1;
			this.player = player;
			this.radius = radius;
			this.world = world;
			this.shape = shape;
		}
		
		public LoaderInfo(NBTTagCompound tag) {
			this.pos = new XYZ(tag.getInteger("X"), tag.getInteger("Y"), tag.getInteger("Z"));
			this.removeTime = tag.getLong("removeTime");
			this.player = tag.getString("player");
			if(this.player.equals(""))
				this.player = null;
			this.isServerOwned = tag.getBoolean("serverOwned");
			shape = Shape.VALUES[tag.getInteger("shape")];
		}
		
		@Override
		public String toString() {
			return "(" + pos + ", " + player + ", r=" + radius + ")";
		}
		
		public String getLogString() {
			return "owner=" + (isServerOwned ? "(nobody)" : player) + ", radius=" + radius + ", shape="+shape;
		}
		
		public long removeTime; // -1 if not waiting to be removed
		public XYZ pos;
		public String player;
		public int radius;
		public WorldInfo world;
		public boolean isServerOwned;
		public Shape shape;
		
		public NBTTagCompound writeNBT() {
			NBTTagCompound tag = new NBTTagCompound();
			tag.setInteger("X", pos.x);
			tag.setInteger("Y", pos.y);
			tag.setInteger("Z", pos.z);
			tag.setLong("removeTime", removeTime);
			tag.setString("player", player == null ? "" : player);
			tag.setBoolean("serverOwned", isServerOwned);
			tag.setInteger("shape", shape.ordinal());
			return tag;
		}

		public Collection<ChunkCoordIntPair> getLoadedChunks() {
			if(radius < 0 || player == null)
				return Collections.emptyList();
			
			int cx = pos.x >> 4;
			int cz = pos.z >> 4;
			int r2 = radius*2+1;
			
			return shape.getLoadedChunks(cx, cz, radius);
		}
	}
	
	private long checkTime = 0;
	
	// stores the number of chunk loaders keeping a chunk loaded
	private HashMap<ChunkCoordIntPair, Integer> loadedChunks = new HashMap<ChunkCoordIntPair, Integer>();
	
	// used for checking quota
	private SetMultimap<String, LoaderInfo> loadersByPlayer = HashMultimap.create();
	
	public void addLoadedChunk(ChunkCoordIntPair ccip) {
		Integer val = loadedChunks.get(ccip);
		if(val != null)
			loadedChunks.put(ccip, val + 1);
		else {
			loadedChunks.put(ccip, 1);
			DimensionalAnchors.cli.addChunk(this, ccip);
		}
		
		World world = getWorld();
		
		if(world != null) // world can be null while loading this WorldInfo from NBT
			DimensionalAnchors.setWorldForceLoaded(world, true);
		
		if(DimensionalAnchors.DEBUG)
			System.out.println("addLoadedChunk("+ccip+")");
	}
	
	public WorldServer getWorld() {
		return worldRef == null ? null : worldRef.get();
	}
	
	public void removeLoadedChunk(ChunkCoordIntPair ccip) {
		World world = getWorld();
		
		Integer val = loadedChunks.get(ccip);
		if(val != null) {
			if(val.intValue() == 1) {
				loadedChunks.remove(ccip);
				DimensionalAnchors.cli.removeChunk(this, ccip);
			} else
				loadedChunks.put(ccip, val - 1);
		}
		
		if(loadedChunks.size() == 0 && world != null)
			DimensionalAnchors.setWorldForceLoaded(world, false);
		
		if(DimensionalAnchors.DEBUG)
			System.out.println("removeLoadedChunk("+ccip+")");
	}
	
	private HashMap<XYZ, LoaderInfo> loaders = new HashMap<XYZ, LoaderInfo>();
	private ArrayList<LoaderInfo> toRemove = new ArrayList<LoaderInfo>();
	
	public WorldInfo(String name) {
		super(name);
	}

	public static WorldInfo get(WorldServer w) {
		String mapname = "ICL-" + w.provider.getSaveFolder();
		File f = w.getSaveHandler().getMapFileFromName(mapname);
		if(!f.getParentFile().exists())
			if(!f.getParentFile().mkdirs())
				DimensionalAnchors.logger.warning("Failed to create directory: " + f.getParentFile());
		
		WorldInfo wi = (WorldInfo)w.mapStorage.loadData(WorldInfo.class, mapname);
		if(wi == null)
		{
			wi = new WorldInfo(mapname);
			wi.worldRef = new WeakReference<WorldServer>(w);
			w.mapStorage.setData(mapname, wi);
		} else {
			wi.worldRef = new WeakReference<WorldServer>(w);
			wi.checkTime = w.getTotalWorldTime() + 40;
			
			wi.checkRemovalTimes(w.getTotalWorldTime());
		}
		return wi;
	}
	
	private void checkRemovalTimes(long curTime) {
		long minTime = curTime - 5;
		long maxTime = curTime + 100;
		for(LoaderInfo li : toRemove) {
			if(li.removeTime != -1 && (li.removeTime < minTime || li.removeTime > maxTime)) {
				li.removeTime = minTime + 5;
			}
		}
	}

	@Override
	public void readFromNBT(NBTTagCompound var1) {
		loaders.clear();
		toRemove.clear();
		{
			NBTTagList list = var1.getTagList("loaders");
			for(int k = 0; k < list.tagCount(); k++) {
				NBTTagCompound c = (NBTTagCompound)list.tagAt(k);
				LoaderInfo loader = new LoaderInfo(c);
				loaders.put(loader.pos, loader);
				if(loader.removeTime != -1)
					toRemove.add(loader);
				if(loader.player != null)
					loadersByPlayer.put(loader.player, loader);
				for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
					addLoadedChunk(ccip);
			}
		}
	}

	@Override
	public void writeToNBT(NBTTagCompound var1) {
		{
			NBTTagList list = new NBTTagList();
			for(LoaderInfo l : loaders.values())
				list.appendTag(l.writeNBT());
			var1.setTag("loaders", list);
		}
	}

	public void delayRemoveLoader(TileChunkLoader tile) {
		World world = getWorld();
		if(world != tile.worldObj)
			throw new IllegalArgumentException("wrong world");
		
		LoaderInfo loader = loaders.get(new XYZ(tile));
		loader.removeTime = world.getTotalWorldTime() + 20; // remove it in one second
		toRemove.add(loader);
		if(DimensionalAnchors.DEBUG)
			System.out.println("Removing "+tile.xCoord+","+tile.yCoord+","+tile.zCoord+" in one second");
		setDirty(true);
	}

	public void addLoader(TileChunkLoader tile) {
		XYZ pos = new XYZ(tile);
		{
			LoaderInfo loader = loaders.get(pos);
			if(loader != null)
				removeLoader(loader);
		}
		LoaderInfo l = tile.getLoaderInfo();
		if(DimensionalAnchors.DEBUG)
			System.out.println("addLoader(" + l + ")");
		loaders.put(pos, l);
		if(l.player != null && !l.isServerOwned)
			loadersByPlayer.put(l.player, l);
		for(ChunkCoordIntPair ccip : l.getLoadedChunks())
			addLoadedChunk(ccip);
		setDirty(true);
	}
	
	private void removeLoader(LoaderInfo loader) {
		if(DimensionalAnchors.DEBUG)
			System.out.println("removeLoader(" + loader + ")");
		loaders.remove(loader.pos);
		toRemove.remove(loader);
		if(loader.player != null)
			loadersByPlayer.remove(loader.player, loader);
		for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
			removeLoadedChunk(ccip);
		setDirty(true);
	}

	public void tick() {
		World world = getWorld();
		if(world == null)
			return;
		
		if(toRemove.size() > 0) {
			LoaderInfo check = toRemove.get(world.rand.nextInt(toRemove.size()));
			if(check.removeTime < world.getTotalWorldTime()) {
				removeLoader(check);
			}
		}
		// Rebuild the loaded chunks and loaders list a short time after loading a world
		if(checkTime != -1 && checkTime < world.getTotalWorldTime()) {
			LinkedList<LoaderInfo> copy = new LinkedList<LoaderInfo>(loaders.values());
			loaders.clear();
			loadedChunks.clear();
			loadersByPlayer.clear();
			checkTime = -1;
			for(LoaderInfo li : copy)
			{
				TileEntity te = world.getBlockTileEntity(li.pos.x, li.pos.y, li.pos.z);
				if(te instanceof TileChunkLoader)
					addLoader((TileChunkLoader)te);
			}
		}
	}
	
	public Collection<? extends ChunkCoordIntPair> getLoadedChunks() {
		return loadedChunks.keySet();
	}
	
	public boolean isChunkForceLoaded(ChunkCoordIntPair pos) {
		return loadedChunks.get(pos) != null;
	}
	
	public int getCurQuota(String player) {
		int chunks = 0;
		for(LoaderInfo l : loadersByPlayer.get(player)) {
			if(l.isServerOwned || l.player == null)
				continue;
			
			// So we don't double-count chunk loaders when they're moved by frames, even
			// though they're still considered as two loaders.
			// Ignores chunk loaders that are about to be removed.
			if(l.removeTime != -1)
				continue;
			
			chunks += l.getLoadedChunks().size();
		}
		return chunks;
	}
	
	
	public void removeLoader(TileChunkLoader tile) {
		removeLoader(loaders.get(new XYZ(tile)));
	}
	public Collection<LoaderInfo> getAllLoaders() {
		return loaders.values();
	}
	
	public String getName() {
		World world = getWorld();
		if(world == null)
			return "<unknown>";
		
		String folder = world.provider.getSaveFolder();
		if(folder == null)
			return "the overworld";
		else
			return "world "+folder;
	}
}
