package immibis.ccperiphs.rfid;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import net.minecraft.src.EntityLiving;
import net.minecraft.src.EntityPlayer;
import net.minecraft.src.ItemStack;
import net.minecraft.src.NBTTagCompound;
import net.minecraft.src.Packet;
import net.minecraft.src.Packet132TileEntityData;

import dan200.computer.api.IComputerAccess;
import dan200.computer.api.IPeripheral;
import immibis.ccperiphs.TilePeriphs;
import immibis.ccperiphs.ImmibisPeripherals;
import immibis.core.api.Dir;

public class TileMagStripe extends TilePeriphs implements IPeripheral {
	
	public static final int MAX_DATA_LENGTH = 100;
	public static final int MAX_LABEL_LENGTH = 20;
	
	private Set<IComputerAccess> computers = new HashSet<IComputerAccess>();
	
	public byte facing;
	
	// Not saved with NBT so this device resets when the attached computer does.
	private String writeData;
	private String writeLabel;
	
	// Visual effects
	public static final int STATE_OFF = 0; // no computer connected
	public static final int STATE_IDLE = 1; // computer connected
	public static final int STATE_READ_WAIT = 2; // computer connected, not writing, insert card now
	public static final int STATE_WRITE_WAIT = 3; // computer connected, writing, insert card now
	public static final int STATE_WRITE = 4; // computer connected, writing
	public int state;
	
	private AtomicBoolean stateUpdateRequired = new AtomicBoolean(true);
	
	private boolean insertCardLight;
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setByte("facing", facing);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		facing = tag.getByte("facing");
	}
	
	private void updateState() {
		int newState;
		if(computers.size() == 0)
			newState = STATE_OFF;
		else {
			if(writeData != null)
				newState = insertCardLight ? STATE_WRITE_WAIT : STATE_WRITE;
			else
				newState = insertCardLight ? STATE_READ_WAIT : STATE_IDLE;
		}
		
		if(state != newState) {
			state = newState;
			worldObj.markBlockForUpdate(xCoord, yCoord, zCoord);
		}
	}
	
	@Override
	public Packet getDescriptionPacket() {
		Packet132TileEntityData p = new Packet132TileEntityData();
		p.actionType = (facing & 7) | (state << 4);
		p.isChunkDataPacket = true;
		p.xPosition = xCoord;
		p.yPosition = yCoord;
		p.zPosition = zCoord;
		return p;
	}
	
	@Override
	public void onDataPacket(Packet132TileEntityData p) {
		facing = (byte)(p.actionType & 7);
		state = (p.actionType >> 4) & 15;
	}

	@Override
	public String getType() {
		return "mag card reader";
	}
	
	private static String[] methodNames = {
		"beginWrite",
		"cancelWrite",
		"isWaiting",
		"setInsertCardLight"
	};

	@Override
	public String[] getMethodNames() {
		return methodNames;
	}

	@Override
	public Object[] callMethod(IComputerAccess computer, int method, Object[] arguments) throws Exception {
		switch(method) {
		case 0: // beginWrite
			if(arguments.length != 2)
				return new Object[] {false, "Expected two arguments"};
			if(!(arguments[0] instanceof String && arguments[1] instanceof String))
				return new Object[] {false, "Expected string"};
			if(writeData != null)
				return new Object[] {false, "Already writing"};
			
			writeData = (String)arguments[0];
			writeLabel = (String)arguments[1];
			if(writeData.length() > MAX_DATA_LENGTH) {
				writeData = writeLabel = null;
				return new Object[] {false, "Max data length is "+MAX_DATA_LENGTH+" chars"};
			}
			if(writeLabel.length() > MAX_LABEL_LENGTH) {
				writeData = writeLabel = null;
				return new Object[] {false, "Max label length is "+MAX_LABEL_LENGTH+" chars"};
			}
			
			stateUpdateRequired.set(true);
			
			return new Object[] {true};
		case 1: // cancelWrite
			writeData = null;
			writeLabel = null;
			stateUpdateRequired.set(true);
			break;
		case 2: // isWaiting
			return new Object[] {writeData != null};
		case 3: // setInsertCardLight
			if(arguments.length == 1 && arguments[0] instanceof Boolean) {
				insertCardLight = (boolean)(Boolean)arguments[0];
				stateUpdateRequired.set(true);
			}
			break;
		}
		return new Object[0];
	}

	@Override
	public boolean canAttachToSide(int side) {
		return true;
	}

	@Override
	public void attach(IComputerAccess computer, String computerSide) {
		computers.add(computer);
		stateUpdateRequired.set(true);
	}

	@Override
	public void detach(IComputerAccess computer) {
		computers.remove(computer);
		
		if(computers.size() == 0)
			writeData = null;

		stateUpdateRequired.set(true);
	}
	
	@Override
	public boolean onBlockActivated(EntityPlayer ply) {
		ItemStack h = ply.getCurrentEquippedItem();
		if(h == null || h.itemID != ImmibisPeripherals.itemMagStripe.shiftedIndex)
			return false;
		
		if(writeData != null) {
			if(h.stackTagCompound == null)
				h.stackTagCompound = new NBTTagCompound();
			
			h.stackTagCompound.setString("data", writeData);
			h.stackTagCompound.setString("line1", writeLabel);
			
			for(IComputerAccess c : computers)
				c.queueEvent("mag_write_done");
			
			writeData = null;
			writeLabel = null;
			
			stateUpdateRequired.set(true);
			
		} else {
			String data = "";
			if(h.stackTagCompound != null)
				data = h.stackTagCompound.getString("data");
			
			Object[] args = new Object[] {data};
			
			for(IComputerAccess c : computers)
				c.queueEvent("mag_swipe", args);
		}
		
		return true;
	}
	
	@Override
	public void onPlaced(EntityLiving player, int look) {
		if(look == Dir.PY)
			facing = (byte)Dir.PY;
		else
			facing = (byte)(look ^ 1);
	}

	@Override
	public void updateEntity() {
		super.updateEntity();
		
		if(!worldObj.isRemote && stateUpdateRequired.compareAndSet(true, false))
			updateState();
	}
}
