package net.minecraft.src;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.*;
import java.util.logging.Level;

import net.minecraft.src.forge.IChunkLoadHandler;
import net.minecraft.src.forge.MinecraftForge;
import net.minecraft.src.forge.NetworkMod;

public class mod_KeepTEsLoaded extends NetworkMod {
	
	private Map<World, Set<ChunkCoordIntPair>> keepLoaded = new HashMap<World, Set<ChunkCoordIntPair>>();
	
	private Set<ChunkCoordIntPair> getKeepLoaded(World w) {
		Set<ChunkCoordIntPair> set = keepLoaded.get(w);
		if(set != null)
			return set;
		set = new HashSet<ChunkCoordIntPair>();
		keepLoaded.put(w, set);
		loadLoadedList(w);
		return set;
	}

	@Override
	public boolean clientSideRequired() {
		return false;
	}

	@Override
	public boolean serverSideRequired() {
		return false;
	}

	@Override
	public String getVersion() {
		return "rev1";
	}
	
	public boolean isClassLoaded(String name) {
		try {
			Class.forName(name);
			//System.err.println(name+" exists");
			return true;
		} catch(ClassNotFoundException e) {
			//System.err.println(name+" doesn't exist");
			return false;
		}
	}

	@Override
	public void load() {
		if(isClassLoaded("net.minecraft.src.mod_ImmiChunkLoaders") || isClassLoaded("mod_ImmiChunkLoaders")) {
			ModLoader.getLogger().warning("mod_KeepTEsLoaded will not load with mod_ImmiChunkLoaders installed.");
			return;
		}
		
		MinecraftForge.registerChunkLoadHandler(new IChunkLoadHandler() {
			
			@Override
			public boolean canUpdateEntity(Entity entity) {
				return getKeepLoaded(entity.worldObj).contains(new ChunkCoordIntPair(entity.chunkCoordX, entity.chunkCoordZ));
			}
			
			@Override
			public boolean canUnloadChunk(Chunk chunk) {
				for(TileEntity te : (Collection<TileEntity>)chunk.chunkTileEntityMap.values())
				{
					if(!isExemptedTileEntity(te))
					{
						if(getKeepLoaded(chunk.worldObj).add(new ChunkCoordIntPair(chunk.xPosition, chunk.zPosition)))
							saveLoadedList(chunk.worldObj);
						return false;
					}
				}
				if(getKeepLoaded(chunk.worldObj).remove(new ChunkCoordIntPair(chunk.xPosition, chunk.zPosition)))
					saveLoadedList(chunk.worldObj);
				return true;
			}
			
			@Override
			public void addActiveChunks(World world, Set<ChunkCoordIntPair> chunkList) {
				chunkList.addAll(getKeepLoaded(world));
			}
		});
	}

	protected void saveLoadedList(World world) {
		String filename = "TEForceLoad-" + world.worldProvider.getSaveFolder();
		File f = world.saveHandler.getMapFileFromName(filename);
		DataOutputStream d_out = null;
		FileOutputStream f_out = null;
		try {
			f_out = new FileOutputStream(f);
			d_out = new DataOutputStream(f_out);
			
			Set<ChunkCoordIntPair> set = new HashSet<ChunkCoordIntPair>(getKeepLoaded(world));
			
			d_out.writeInt(set.size());
			
			for(ChunkCoordIntPair p : set) {
				d_out.writeInt(p.chunkXPos);
				d_out.writeInt(p.chunkZPosition);
			}
		} catch(IOException e) {
			ModLoader.getLogger().log(Level.WARNING, "Exception while saving "+filename+".dat");
		} finally {
			try {
				if(d_out != null)
					d_out.close();
				else if(f_out != null)
					f_out.close();
			} catch(Exception e) {
			}
		}
	}
	
	protected void loadLoadedList(World world) {
		String filename = "TEForceLoad-" + world.worldProvider.getSaveFolder();
		File f = world.saveHandler.getMapFileFromName(filename);
		if(!f.exists())
			return;
		DataInputStream d_in = null;
		FileInputStream f_in = null;
		try {
			f_in = new FileInputStream(f);
			d_in = new DataInputStream(f_in);
			
			Set<ChunkCoordIntPair> set = new HashSet<ChunkCoordIntPair>();
			
			int count = d_in.readInt();
			for(int k = 0; k < count; k++) {
				int x = d_in.readInt();
				int z = d_in.readInt();
				set.add(new ChunkCoordIntPair(x, z));
			}
			
			keepLoaded.put(world, set);
		} catch(IOException e) {
			ModLoader.throwException("Loading "+filename, e);
		} finally {
			try {
				if(d_in != null)
					d_in.close();
				else if(f_in != null)
					f_in.close();
			} catch(Exception e) {
			}
		}
	}

	protected boolean isExemptedTileEntity(TileEntity te) {
		Class<? extends TileEntity> cl = te.getClass();
		return
			cl == TileEntityChest.class ||
			cl == TileEntitySign.class ||
			cl == TileEntityEndPortal.class ||
			cl == TileEntityMobSpawner.class ||
			cl == TileEntityRecordPlayer.class;
	}

}
