package mods.immibis.recycling;

import ic2.api.recipe.RecipeInputItemStack;
import ic2.api.recipe.RecipeOutput;
import ic2.api.recipe.Recipes;

import java.util.ArrayList;
import java.util.List;

import thermalexpansion.util.crafting.PulverizerManager;
import thermalexpansion.util.crafting.PulverizerManager.RecipePulverizer;
import mods.immibis.core.api.FMLModInfo;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Items;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.CraftingManager;
import net.minecraft.item.crafting.FurnaceRecipes;
import net.minecraftforge.common.config.Configuration;
import net.minecraftforge.common.config.Property;
import net.minecraftforge.oredict.OreDictionary;
import cpw.mods.fml.common.Loader;
import cpw.mods.fml.common.Mod;
import cpw.mods.fml.common.Mod.EventHandler;
import cpw.mods.fml.common.event.FMLPostInitializationEvent;
import cpw.mods.fml.common.event.FMLPreInitializationEvent;

@Mod(modid="SimpleRecycling", name="Simple Recycling", version="59.0.0")
@FMLModInfo(
	url="http://www.minecraftforum.net/topic/1001131-110-immibiss-mods-smp/",
	description="Allows you to turn tools back into ingots or dusts.",
	authors="immibis"
	)
public class SimpleRecycling {
	
	private static final boolean DUMP_RECIPES = Boolean.getBoolean("mods.immibis.recycling.dumpRecipes");
	
	List<BlockMetaPair> ingots = new ArrayList<BlockMetaPair>();
	List<BlockMetaPair> gems = new ArrayList<BlockMetaPair>();
	List<ToolMaterialEntry> configuredToolMaterials = new ArrayList<ToolMaterialEntry>();
	boolean useCache, doDiscoverTools;
	boolean smelt, macerate;
	IC2 ic2;
	TE4 te4;
	
	private static class IC2 {
		public ItemStack getMacerated(ItemStack stack) {
			RecipeOutput out = Recipes.macerator.getOutputFor(stack, false);
			return out == null ? null : out.items.get(0);
		}

		public void addMaceratorRecipe(ItemStack in, ItemStack out) {
			Recipes.macerator.addRecipe(new RecipeInputItemStack(in), null, out);
		}
	}
	
	private static class TE4 {
		public void addPulverizerRecipeIfNotExists(ItemStack craftedItem, ItemStack ingotStack, int numIngots) {
			if(PulverizerManager.getRecipe(craftedItem) != null)
				return; // item is already pulverizable
			
			RecipePulverizer recipe = PulverizerManager.getRecipe(ingotStack);
			if(recipe == null)
				return; // material is not pulverizable
			
			ItemStack primaryOutput = ItemStack.copyItemStack(recipe.getPrimaryOutput());
			ItemStack secondaryOutput = ItemStack.copyItemStack(recipe.getSecondaryOutput());
			
			int energy = recipe.getEnergy() * numIngots;
			
			int secondaryChance = 0;
			if(secondaryOutput != null) {
				secondaryChance = secondaryOutput.stackSize * recipe.getSecondaryOutputChance() * numIngots;
				secondaryOutput.stackSize = 1;
			}
			
			if(primaryOutput != null) primaryOutput.stackSize *= numIngots;
			
			if(DUMP_RECIPES)
				System.err.println("[Simple Recycling] adding pulverizer recipe for "+craftedItem+" -> "+primaryOutput+" + "+secondaryChance+"% chance of "+secondaryOutput+", using "+energy+" RF");
			if(!PulverizerManager.addRecipe(energy, craftedItem, primaryOutput, secondaryOutput, secondaryChance, false)) {
				System.err.println("[Simple Recycling] notice: failed to add pulverizer recipe for "+craftedItem+" -> "+primaryOutput+" + "+secondaryChance+"% chance of "+secondaryOutput+", using "+energy+" RF");
			}
		}
	}
	
	private static class BlockMetaPair {
		public String id;
		public int data;
		
		public BlockMetaPair(String blockID, int k) {
			this.id = blockID;
			this.data = k;
		}

		public static BlockMetaPair parse(String s) {
			String[] a = s.split("@");
			if(a.length != 2)
				throw new NumberFormatException("Not a valid id@meta pair: " + s);
			return new BlockMetaPair(a[0], Integer.parseInt(a[1]));
		}
		
		public ItemStack toItemStack() {
			Item item = (Item)Item.itemRegistry.getObject(id);
			if(item == null)
				throw new RuntimeException("no such item: "+id);
			return new ItemStack(item, 1, data);
		}
	}
	
	private static class ToolMaterialEntry {
		public BlockMetaPair tool, mat;
		public int matAmt;
		public boolean smeltable;
		
		public ToolMaterialEntry(String toolID, int toolMeta, String matID, int matMeta, int matAmt, boolean smeltable) {
			this.tool = new BlockMetaPair(toolID, toolMeta);
			this.mat = new BlockMetaPair(matID, matMeta);
			this.matAmt = matAmt;
			this.smeltable = smeltable;
		}

		public static ToolMaterialEntry parse(String s) {
			String[] a = s.split("@");
			try {
				if(a.length != 6)
					throw new NumberFormatException();
				if(!a[5].equals("Y") && !a[5].equals("N"))
					throw new NumberFormatException();
				return new ToolMaterialEntry(a[0], Integer.parseInt(a[1]), a[2], Integer.parseInt(a[3]), Integer.parseInt(a[4]), a[5].equals("Y")); 
			} catch(NumberFormatException e) {
				throw new RuntimeException("Not a valid tool entry (format is toolID@toolMeta@materialID@materialMeta@materialAmount@smeltable where smeltable is Y or N): " + s);
			}
		}
	}
	
	@EventHandler
	public void onPreInit(FMLPreInitializationEvent evt) {
		Configuration c = new Configuration(evt.getSuggestedConfigurationFile());
		c.load();
		
		boolean dirty = false;
		
		String COMMENT;
		
		Property ingotsProp = c.get(Configuration.CATEGORY_GENERAL, "ingots", "");
		COMMENT = "List of id:meta pairs, separated by semicolons. Example: 9001:2;4567:100";
		if(!COMMENT.equals(ingotsProp.comment)) {
			ingotsProp.comment = COMMENT;
			dirty = true;
		}
		
		Property gemsProp = c.get(Configuration.CATEGORY_GENERAL, "gems", "");
		COMMENT = "List of id:meta pairs, separated by semicolons. Example: 9001:2;4567:100";
		if(!COMMENT.equals(gemsProp.comment)) {
			gemsProp.comment = COMMENT;
			dirty = true;
		}
		
		Property toolsProp = c.get(Configuration.CATEGORY_GENERAL, "tools", "");
		COMMENT = "Semicolon-separated list of toolID:toolMeta:materialID:materialMeta:materialAmount:smeltable. Example: 9001:0:1:0:2:Y";
		if(!COMMENT.equals(toolsProp.comment)) {
			toolsProp.comment = COMMENT;
			dirty = true;
		}
		
		
		for(String p : ingotsProp.getString().split(";"))
			if(!p.equals(""))
				ingots.add(BlockMetaPair.parse(p));
		
		for(String p : gemsProp.getString().split(";"))
			if(!p.equals(""))
				gems.add(BlockMetaPair.parse(p));
		
		for(String p : toolsProp.getString().split(";"))
			if(!p.equals(""))
				configuredToolMaterials.add(ToolMaterialEntry.parse(p));
		
		smelt = c.get(Configuration.CATEGORY_GENERAL, "allowSmelting", true).getBoolean(true);
		macerate = c.get(Configuration.CATEGORY_GENERAL, "allowMacerating", true).getBoolean(true);
		
		if(dirty || c.hasChanged())
			c.save();
	}
	
	@EventHandler
	public void onPostInit(FMLPostInitializationEvent evt) {
		InventoryCrafting ic = new InventoryCrafting(new Container() {
			@Override
			public boolean canInteractWith(EntityPlayer entityplayer) {
				return false;
			}
		}, 3, 3);
		
		if(Loader.isModLoaded("IC2"))
			ic2 = new IC2();
		
		if(Loader.isModLoaded("ThermalExpansion"))
			te4 = new TE4();
		
		long startTime = System.nanoTime();
		
		for(BlockMetaPair bmp : ingots) register(ic, true, bmp.toItemStack());
		for(BlockMetaPair bmp : gems) register(ic, true, bmp.toItemStack());
		
		for(String oreName : OreDictionary.getOreNames()) {
			List<ItemStack> ores = OreDictionary.getOres(oreName);
			if(ores.size() == 0)
				continue;
			
			if(oreName.startsWith("ingot"))
				register(ic, true, ores.get(0));
			else
				register(ic, false, ores.get(0));
		}
		
		register(ic, false, new ItemStack(Items.diamond));
		
		for(ToolMaterialEntry t : configuredToolMaterials)
			register(t.smeltable, t.tool.toItemStack(), t.mat.toItemStack(), t.matAmt);
		
		long elapsed = (System.nanoTime() - startTime);
		System.out.println("[Simple Recycling] Loaded. Time taken: "+(elapsed/1000000)+" ms");
	}
	
	private void register(InventoryCrafting ic, boolean canSmelt, ItemStack ingotStack, String pattern) {
		int numIngots = 0;
		for(int k = 0; k < 9; k++)
		{
			char c = pattern.charAt(k);
			if(c == '#') {
				ic.setInventorySlotContents(k, ingotStack);
				numIngots++;
			} else if(c == ' ')
				ic.setInventorySlotContents(k, null);
			else if(c == '|')
				ic.setInventorySlotContents(k, new ItemStack(Items.stick));
			else
				throw new RuntimeException("pattern char "+c);
		}
		
		ItemStack craftedItem;
		try {
			craftedItem = CraftingManager.getInstance().findMatchingRecipe(ic, null);
		} catch(Exception e) {
			new Exception("[Simple Recycling] Caught exception looking up recipe for item '"+ingotStack+"' pattern '"+pattern+"'", e).printStackTrace();
			return;
		}
		
		if(craftedItem == null)
			return;
		
		register(canSmelt, craftedItem, ingotStack, numIngots);
	}
	
	private void register(boolean canSmelt, ItemStack craftedItem, ItemStack ingotStack, int numIngots) {
		// add smelting recipe, if smelting recipes are enabled, this item is smeltable, and the recipe doesn't already exist
		if(smelt && canSmelt && FurnaceRecipes.smelting().getSmeltingResult(craftedItem) == null) {
			ItemStack smeltingOutput = ingotStack.copy();
			smeltingOutput.stackSize = numIngots;
			
			//if(craftedItem.getItem().isDamageable())
			//	FurnaceRecipes.smelting().addSmelting(craftedItem, smeltingOutput, 0);
			//else
				FurnaceRecipes.smelting().func_151394_a(craftedItem, smeltingOutput, 0);
		}
		
		// add macerator recipe, if macerating recipes are enabled, this item is maceratable, and the recipe doesn't already exist
		if(macerate && ic2 != null && ic2.getMacerated(craftedItem) == null) {
			ItemStack maceratingOutput = ic2.getMacerated(ingotStack);
			if(maceratingOutput != null) {
				maceratingOutput = maceratingOutput.copy();
				maceratingOutput.stackSize = numIngots;
				
				ic2.addMaceratorRecipe(craftedItem, maceratingOutput);
			}
		}
		
		if(macerate && te4 != null)
			te4.addPulverizerRecipeIfNotExists(craftedItem, ingotStack, numIngots);
	}
	
	private void register(InventoryCrafting ic, boolean canSmelt, ItemStack ingotStack) {
		register(ic, canSmelt, ingotStack, "### |  | "); // pickaxe
		register(ic, canSmelt, ingotStack, " #  |  | "); // shovel
		register(ic, canSmelt, ingotStack, "## #|  | "); // axe (1)
		register(ic, canSmelt, ingotStack, " ## |# | "); // axe (2)
		register(ic, canSmelt, ingotStack, " #  #  | "); // sword
		register(ic, canSmelt, ingotStack, "##  |  | "); // hoe (1)
		register(ic, canSmelt, ingotStack, " ## |  | "); // hoe (2)
		register(ic, canSmelt, ingotStack, "     # # "); // shears (1)
		register(ic, canSmelt, ingotStack, "    #   #"); // shears (2)
		register(ic, canSmelt, ingotStack, " #   #|# "); // sickle
		register(ic, canSmelt, ingotStack, "#### ## #"); // leggings
		register(ic, canSmelt, ingotStack, "#### #   "); // helmet
		register(ic, canSmelt, ingotStack, "# #######"); // chestplate
		register(ic, canSmelt, ingotStack, "# ## #   "); // boots
	}
}
