package immibis.chunkloader;

import net.minecraft.src.*;
import immibis.core.TileCombined;

public class TileChunkLoader extends TileCombined {
	public WorldInfo worldInfo;
	
	public String owner;
	public int radius = 0;
	public boolean isServerOwned = false;
	
	private boolean wasPreviouslyInvalidated = false;
	
	public boolean canUpdate() {
		return false;
	}

	public TileChunkLoader() {
	}
	
	@Override
	public void writeToNBT(NBTTagCompound tag) {
		super.writeToNBT(tag);
		tag.setString("owner", owner == null ? "" : owner);
		tag.setInteger("radius", radius);
		tag.setBoolean("serverOwned", isServerOwned);
	}
	
	@Override
	public void readFromNBT(NBTTagCompound tag) {
		super.readFromNBT(tag);
		owner = tag.getString("owner");
		if(owner != null && owner.equals(""))
			owner = null;
		radius = tag.getInteger("radius");
		isServerOwned = tag.getBoolean("serverOwned");
	}
	
	@Override
	public boolean onBlockActivated(EntityPlayer ply) {
		ply.openGui(Main.instance, Main.GUI_CHUNKLOADER, worldObj, xCoord, yCoord, zCoord);
		return true;
	}
	
	@Override
	public void validate() {
		super.validate();
		worldInfo = Main.instance.getWorld(worldObj);
		worldInfo.addLoader(this);
		if(wasPreviouslyInvalidated)
			loadChunks();
		wasPreviouslyInvalidated = false;
		if(owner != null)
			Logging.onValidated(getLoaderInfo(), "<unknown>");
	}
	@Override
	public void invalidate() {
		super.invalidate();
		worldInfo.delayRemoveLoader(this);
		if(owner != null)
			Logging.onInvalidated(getLoaderInfo(), "<unknown>");
		wasPreviouslyInvalidated = true;
	}
	
	private void loaderChanged() {
		worldInfo.readdLoader(this);
	}
	
	public void loaderChanged(String byWho) {
		loaderChanged(byWho, false);
	}
	
	public void loaderChanged(String byWho, boolean justAdded) {
		worldInfo.readdLoader(this);
		if(justAdded)
			Logging.onAdded(getLoaderInfo(), byWho);
		else
			Logging.onChanged(getLoaderInfo(), byWho);
	}
	
	void loadChunks() {
		IChunkProvider provider = worldObj.getChunkProvider();
		for(ChunkCoordIntPair ccip : getLoaderInfo().getLoadedChunks())
			/* $if mc < 1.3$
			if(provider.provideChunk(ccip.chunkXPos, ccip.chunkZPosition) instanceof EmptyChunk)
				provider.loadChunk(ccip.chunkXPos, ccip.chunkZPosition);
			$else$ */
			if(provider.provideChunk(ccip.chunkXPos, ccip.chunkZPos) instanceof EmptyChunk)
				provider.loadChunk(ccip.chunkXPos, ccip.chunkZPos);
			/* $endif$ */
	}
	
	public void limitRadius() {
		if(owner == null) {
			radius = -1;
			return;
		}
		
		int max = Main.instance.getMaxQuota(owner);
		
		if(max == -2)
			return;
		
		while(radius >= 0 && Main.instance.getCurQuota(owner) + (2*radius+1)*(2*radius+1) > max) {
			radius--;
			loaderChanged("<enforcing quota limit for "+owner+">");
		}
	}

	public WorldInfo.LoaderInfo getLoaderInfo() {
		WorldInfo.LoaderInfo li = new WorldInfo.LoaderInfo(new WorldInfo.XYZ(this), worldInfo, owner, radius);
		li.isServerOwned = isServerOwned;
		return li;
	}
}
