package immibis.chunkloader;

import java.util.*;

import net.minecraft.src.*;

public class WorldInfo extends WorldSavedData {
	
	private World world;
	
	public static class XYZ {
		int x, y, z;
		public XYZ(int x, int y, int z) {
			this.x = x;
			this.y = y;
			this.z = z;
		}
		public int hashCode() {
			return x * 23434 + y * 2351321 + z;
		}
		public boolean equals(Object o) {
			try {
				XYZ _o = (XYZ)o;
				return x == _o.x && y == _o.y && z == _o.z;
			} catch(ClassCastException e) {
				return false;
			}
		}
		public XYZ(TileEntity tile) {
			this(tile.xCoord, tile.yCoord, tile.zCoord);
		}
		
		public String toString() {
			return "["+x+","+y+","+z+"]";
		}
	}
	
	public static class LoaderInfo {
		public LoaderInfo(XYZ pos, WorldInfo world, String player, int radius) {
			this.pos = pos;
			this.removeTime = -1;
			this.player = player;
			this.radius = radius;
			this.world = world;
		}
		
		public LoaderInfo(NBTTagCompound tag) {
			this.pos = new XYZ(tag.getInteger("X"), tag.getInteger("Y"), tag.getInteger("Z"));
			this.removeTime = tag.getLong("removeTime");
			this.player = tag.getString("player");
			if(this.player.equals(""))
				this.player = null;
			this.isServerOwned = tag.getBoolean("serverOwned");
		}
		
		@Override
		public String toString() {
			return "(" + pos + ", " + player + ", r=" + radius + ")";
		}
		
		public String getLogString() {
			return "owner=" + (isServerOwned ? "(nobody)" : player) + ", radius=" + radius;
		}
		
		public long removeTime; // -1 if not waiting to be removed
		public XYZ pos;
		public String player;
		public int radius;
		public WorldInfo world;
		public boolean isServerOwned;
		
		public NBTTagCompound writeNBT() {
			NBTTagCompound tag = new NBTTagCompound();
			tag.setInteger("X", pos.x);
			tag.setInteger("Y", pos.y);
			tag.setInteger("Z", pos.z);
			tag.setLong("removeTime", removeTime);
			tag.setString("player", player == null ? "" : player);
			tag.setBoolean("serverOwned", isServerOwned);
			return tag;
		}

		public Collection<ChunkCoordIntPair> getLoadedChunks() {
			if(radius < 0 || player == null)
				return Collections.emptyList();
			
			int cx = pos.x >> 4;
			int cz = pos.z >> 4;
			int r2 = radius*2+1;
			ChunkCoordIntPair[] a = new ChunkCoordIntPair[r2 * r2];
			int pos = 0;
			for(int dx = -radius; dx <= radius; dx++) {
				for(int dz = -radius; dz <= radius; dz++) {
					a[pos++] = new ChunkCoordIntPair(cx + dx, cz + dz);
				}
			}
			return Arrays.asList(a);
		}
	}
	
	private long checkTime = 0;
	
	// stores the number of chunk loaders keeping a chunk loaded
	private HashMap<ChunkCoordIntPair, Integer> loadedChunks = new HashMap<ChunkCoordIntPair, Integer>();
	
	private HashMap<String, Integer> curQuota = new HashMap<String, Integer>();
	
	public void addLoadedChunk(ChunkCoordIntPair ccip, String player) {
		Integer val = loadedChunks.get(ccip);
		if(val != null)
			loadedChunks.put(ccip, val + 1);
		else
			loadedChunks.put(ccip, 1);
		
		if(player != null)
		{
			val = curQuota.get(player);
			if(val != null)
				curQuota.put(player, val + 1);
			else
				curQuota.put(player, 1);
		}
		
		if(Main.DEBUG)
			System.out.println("addLoadedChunk("+ccip+","+player+")");
	}
	public void removeLoadedChunk(ChunkCoordIntPair ccip, String player) {
		Integer val = loadedChunks.get(ccip);
		if(val != null) {
			if(val.intValue() == 1)
				loadedChunks.remove(ccip);
			else
				loadedChunks.put(ccip, val - 1);
		}
		
		if(player != null) {
			val = curQuota.get(player);
			if(val != null) {
				if(val.intValue() == 1)
					curQuota.remove(player);
				else
					curQuota.put(player, val - 1);
			}
		}
		
		if(Main.DEBUG)
			System.out.println("removeLoadedChunk("+ccip+","+player+")");
	}
	
	private HashMap<XYZ, LoaderInfo> loaders = new HashMap<XYZ, LoaderInfo>();
	private ArrayList<LoaderInfo> toRemove = new ArrayList<LoaderInfo>();
	
	public WorldInfo(String name) {
		super(name);
	}

	public static WorldInfo get(World w) {
		/* $if mc < 1.3$
		String mapname = "ICL-" + w.worldProvider.getSaveFolder();
		$else$ */
		String mapname = "ICL-" + w.provider.getSaveFolder();
		/* $endif$ */
		WorldInfo wi = (WorldInfo)w.mapStorage.loadData(WorldInfo.class, mapname);
		if(wi == null)
		{
			wi = new WorldInfo(mapname);
			wi.world = w;
			w.mapStorage.setData(mapname, wi);
		} else {
			wi.world = w;
			wi.checkTime = w.getWorldTime() + 40;
		}
		return wi;
	}

	@Override
	public void readFromNBT(NBTTagCompound var1) {
		loaders.clear();
		toRemove.clear();
		{
			NBTTagList list = var1.getTagList("loaders");
			for(int k = 0; k < list.tagCount(); k++) {
				NBTTagCompound c = (NBTTagCompound)list.tagAt(k);
				LoaderInfo loader = new LoaderInfo(c);
				loaders.put(loader.pos, loader);
				if(loader.removeTime != -1)
					toRemove.add(loader);
				for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
					addLoadedChunk(ccip, loader.player);
			}
		}
	}

	@Override
	public void writeToNBT(NBTTagCompound var1) {
		{
			NBTTagList list = new NBTTagList();
			for(LoaderInfo l : loaders.values())
				list.appendTag(l.writeNBT());
			var1.setTag("loaders", list);
		}
	}

	public void delayRemoveLoader(TileChunkLoader tile) {
		LoaderInfo loader = loaders.get(new XYZ(tile));
		loader.removeTime = world.getWorldTime() + 20; // remove it in one second
		toRemove.add(loader);
		if(Main.DEBUG)
			System.out.println("Removing "+tile.xCoord+","+tile.yCoord+","+tile.zCoord+" in one second");
		setDirty(true);
	}

	public void addLoader(TileChunkLoader tile) {
		XYZ pos = new XYZ(tile);
		{
			LoaderInfo loader = loaders.get(pos);
			if(loader != null)
				removeLoader(loader);
		}
		LoaderInfo l = tile.getLoaderInfo();
		if(Main.DEBUG)
			System.out.println("addLoader(" + l + ")");
		loaders.put(pos, l);
		for(ChunkCoordIntPair ccip : l.getLoadedChunks())
			addLoadedChunk(ccip, l.isServerOwned ? null : l.player);
		setDirty(true);
	}
	
	private void removeLoader(LoaderInfo loader) {
		if(Main.DEBUG)
			System.out.println("removeLoader(" + loader + ")");
		loaders.remove(loader.pos);
		toRemove.remove(loader);
		for(ChunkCoordIntPair ccip : loader.getLoadedChunks())
			removeLoadedChunk(ccip, loader.isServerOwned ? null : loader.player);
		setDirty(true);
	}

	public void tick() {
		if(toRemove.size() > 0) {
			LoaderInfo check = toRemove.get(world.rand.nextInt(toRemove.size()));
			if(check.removeTime < world.getWorldTime()) {
				removeLoader(check);
			}
		}
		// Rebuild the loaded chunks and loaders list a short time after loading a world
		if(checkTime != -1 && checkTime < world.getWorldTime()) {
			LinkedList<LoaderInfo> copy = new LinkedList<LoaderInfo>(loaders.values());
			loaders.clear();
			loadedChunks.clear();
			curQuota.clear();
			checkTime = -1;
			for(LoaderInfo li : copy)
			{
				TileEntity te = world.getBlockTileEntity(li.pos.x, li.pos.y, li.pos.z);
				if(te instanceof TileChunkLoader)
					addLoader((TileChunkLoader)te);
			}
		}
	}
	
	public Collection<? extends ChunkCoordIntPair> getLoadedChunks() {
		return loadedChunks.keySet();
	}
	
	public boolean isChunkForceLoaded(ChunkCoordIntPair pos) {
		return loadedChunks.get(pos) != null;
	}
	
	public int getCurQuota(String player) {
		Integer val = curQuota.get(player);
		return val == null ? 0 : val;
	}
	public void readdLoader(TileChunkLoader tile) {
		removeLoader(loaders.get(new XYZ(tile)));
		addLoader(tile);
	}
	public Collection<LoaderInfo> getAllLoaders() {
		return loaders.values();
	}
	
	public String getName() {
		/* $if mc < 1.3$
		String folder = world.worldProvider.getSaveFolder();
		$else$ */
		String folder = world.provider.getSaveFolder();
		/* $endif$ */
		if(folder == null)
			return "the overworld";
		else
			return "world "+folder;
	}
}
